/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.transportconstraintprofile;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.TransportConstraint;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of constraintList.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultConstraintList extends InnerModelObject 
        implements ConstraintList, MultiInstanceObject<ConstraintListKeys> {
    protected Class<? extends TransportConstraint> constraintType;
    protected String constraintOpaqueValue;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public Class<? extends TransportConstraint> constraintType() {
        return constraintType;
    }

    @Override
    public String constraintOpaqueValue() {
        return constraintOpaqueValue;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void constraintType(Class<? extends TransportConstraint> constraintType) {
        valueLeafFlags.set(LeafIdentifier.CONSTRAINTTYPE.getLeafIndex());
        this.constraintType = constraintType;
    }

    @Override
    public void constraintOpaqueValue(String constraintOpaqueValue) {
        valueLeafFlags.set(LeafIdentifier.CONSTRAINTOPAQUEVALUE.getLeafIndex());
        this.constraintOpaqueValue = constraintOpaqueValue;
    }

    @Override
    public int hashCode() {
        return Objects.hash(constraintType, constraintOpaqueValue, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultConstraintList) {
            DefaultConstraintList other = (DefaultConstraintList) obj;
            return
                Objects.deepEquals(constraintType, other.constraintType) &&
                Objects.deepEquals(constraintOpaqueValue, other.constraintOpaqueValue) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("constraintType=" + constraintType)
            .add("constraintOpaqueValue=" + constraintOpaqueValue)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultConstraintList.
     */
    public DefaultConstraintList() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}