/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.sitevpnpolicy.vpnpolicylist;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.SvcId;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.sitevpnpolicy.vpnpolicylist.vpnpolicy.Entries;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of vpnPolicy.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultVpnPolicy extends InnerModelObject 
        implements VpnPolicy, MultiInstanceObject<VpnPolicyKeys> {
    protected SvcId vpnPolicyId;
    protected BitSet valueLeafFlags = new BitSet();
    protected List<Entries> entries;

    @Override
    public SvcId vpnPolicyId() {
        return vpnPolicyId;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public List<Entries> entries() {
        return entries;
    }

    @Override
    public void vpnPolicyId(SvcId vpnPolicyId) {
        valueLeafFlags.set(LeafIdentifier.VPNPOLICYID.getLeafIndex());
        this.vpnPolicyId = vpnPolicyId;
    }

    @Override
    public void entries(List<Entries> entries) {
        this.entries = entries;
    }

    @Override
    public void addToEntries(Entries addTo) {
        if (entries == null) {
            entries = new ArrayList<>();
        }
        entries.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(vpnPolicyId, valueLeafFlags, entries);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultVpnPolicy) {
            DefaultVpnPolicy other = (DefaultVpnPolicy) obj;
            return
                Objects.deepEquals(vpnPolicyId, other.vpnPolicyId) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(entries, other.entries);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("vpnPolicyId=" + vpnPolicyId)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("entries=" + entries)
            .toString();
    }


    /**
     * Creates an instance of defaultVpnPolicy.
     */
    public DefaultVpnPolicy() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}