/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.siteattachmentipconnection.ipconnection.ipv4;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtietfinettypes.rev20130715.yrtietfinettypes.Ipv4Address;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of addresses.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultAddresses extends InnerModelObject implements Addresses {
    protected Ipv4Address providerAddress;
    protected Ipv4Address customerAddress;
    protected short mask;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public Ipv4Address providerAddress() {
        return providerAddress;
    }

    @Override
    public Ipv4Address customerAddress() {
        return customerAddress;
    }

    @Override
    public short mask() {
        return mask;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void providerAddress(Ipv4Address providerAddress) {
        valueLeafFlags.set(LeafIdentifier.PROVIDERADDRESS.getLeafIndex());
        this.providerAddress = providerAddress;
    }

    @Override
    public void customerAddress(Ipv4Address customerAddress) {
        valueLeafFlags.set(LeafIdentifier.CUSTOMERADDRESS.getLeafIndex());
        this.customerAddress = customerAddress;
    }

    @Override
    public void mask(short mask) {
        valueLeafFlags.set(LeafIdentifier.MASK.getLeafIndex());
        this.mask = mask;
    }

    @Override
    public int hashCode() {
        return Objects.hash(providerAddress, customerAddress, mask, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultAddresses) {
            DefaultAddresses other = (DefaultAddresses) obj;
            return
                Objects.deepEquals(providerAddress, other.providerAddress) &&
                Objects.deepEquals(customerAddress, other.customerAddress) &&
                Objects.deepEquals(mask, other.mask) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("providerAddress=" + providerAddress)
            .add("customerAddress=" + customerAddress)
            .add("mask=" + mask)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultAddresses.
     */
    public DefaultAddresses() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}