/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtietftetopology.rev20160317.yrtietftetopology.telinkconfig.bundlestacklevel.bundle.bundledlinks;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of bundledLink.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultBundledLink extends InnerModelObject 
        implements BundledLink, MultiInstanceObject<BundledLinkKeys> {
    protected long sequence;
    protected Object srcTpRef;
    protected Object desTpRef;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public long sequence() {
        return sequence;
    }

    @Override
    public Object srcTpRef() {
        return srcTpRef;
    }

    @Override
    public Object desTpRef() {
        return desTpRef;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void sequence(long sequence) {
        valueLeafFlags.set(LeafIdentifier.SEQUENCE.getLeafIndex());
        this.sequence = sequence;
    }

    @Override
    public void srcTpRef(Object srcTpRef) {
        valueLeafFlags.set(LeafIdentifier.SRCTPREF.getLeafIndex());
        this.srcTpRef = srcTpRef;
    }

    @Override
    public void desTpRef(Object desTpRef) {
        valueLeafFlags.set(LeafIdentifier.DESTPREF.getLeafIndex());
        this.desTpRef = desTpRef;
    }

    @Override
    public int hashCode() {
        return Objects.hash(sequence, srcTpRef, desTpRef, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultBundledLink) {
            DefaultBundledLink other = (DefaultBundledLink) obj;
            return
                Objects.equals(sequence, other.sequence) &&
                Objects.equals(srcTpRef, other.srcTpRef) &&
                Objects.equals(desTpRef, other.desTpRef) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("sequence=" + sequence)
            .add("srcTpRef=" + srcTpRef)
            .add("desTpRef=" + desTpRef)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultBundledLink.
     */
    public DefaultBundledLink() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}