/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.sample.sample;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.sample.sample.top.YangAutoPrefixInterface;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of top.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultTop extends InnerModelObject implements Top {
    protected String mtu;
    protected List<String> color;
    protected BitSet valueLeafFlags = new BitSet();
    protected List<YangAutoPrefixInterface> yangAutoPrefixInterface;

    @Override
    public String mtu() {
        return mtu;
    }

    @Override
    public List<String> color() {
        return color;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public List<YangAutoPrefixInterface> yangAutoPrefixInterface() {
        return yangAutoPrefixInterface;
    }

    @Override
    public void mtu(String mtu) {
        valueLeafFlags.set(LeafIdentifier.MTU.getLeafIndex());
        this.mtu = mtu;
    }

    @Override
    public void color(List<String> color) {
        valueLeafFlags.set(LeafIdentifier.COLOR.getLeafIndex());
        this.color = color;
    }

    @Override
    public void yangAutoPrefixInterface(List<YangAutoPrefixInterface> yangAutoPrefixInterface) {
        this.yangAutoPrefixInterface = yangAutoPrefixInterface;
    }

    @Override
    public void addToColor(String addTo) {
        if (color == null) {
            color = new ArrayList<>();
        }
        color.add(addTo);
    }


    @Override
    public void addToYangAutoPrefixInterface(YangAutoPrefixInterface addTo) {
        if (yangAutoPrefixInterface == null) {
            yangAutoPrefixInterface = new ArrayList<>();
        }
        yangAutoPrefixInterface.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(mtu, color, valueLeafFlags, yangAutoPrefixInterface);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultTop) {
            DefaultTop other = (DefaultTop) obj;
            return
                Objects.equals(mtu, other.mtu) &&
                Objects.equals(color, other.color) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.equals(yangAutoPrefixInterface, other.yangAutoPrefixInterface);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("mtu=" + mtu)
            .add("color=" + color)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("yangAutoPrefixInterface=" + yangAutoPrefixInterface)
            .toString();
    }


    /**
     * Creates an instance of defaultTop.
     */
    public DefaultTop() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}