/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.sample.sample;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.sample.sample.l1.C1;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of l1.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultL1 extends InnerModelObject 
        implements L1, MultiInstanceObject<L1Keys> {
    protected String leaf3;
    protected List<String> leaf4;
    protected BitSet valueLeafFlags = new BitSet();
    protected C1 c1;

    @Override
    public String leaf3() {
        return leaf3;
    }

    @Override
    public List<String> leaf4() {
        return leaf4;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public C1 c1() {
        return c1;
    }

    @Override
    public void leaf3(String leaf3) {
        valueLeafFlags.set(LeafIdentifier.LEAF3.getLeafIndex());
        this.leaf3 = leaf3;
    }

    @Override
    public void leaf4(List<String> leaf4) {
        valueLeafFlags.set(LeafIdentifier.LEAF4.getLeafIndex());
        this.leaf4 = leaf4;
    }

    @Override
    public void c1(C1 c1) {
        this.c1 = c1;
    }

    @Override
    public void addToLeaf4(String addTo) {
        if (leaf4 == null) {
            leaf4 = new ArrayList<>();
        }
        leaf4.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(leaf3, leaf4, valueLeafFlags, c1);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultL1) {
            DefaultL1 other = (DefaultL1) obj;
            return
                Objects.equals(leaf3, other.leaf3) &&
                Objects.equals(leaf4, other.leaf4) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.equals(c1, other.c1);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("leaf3=" + leaf3)
            .add("leaf4=" + leaf4)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("c1=" + c1)
            .toString();
    }


    /**
     * Creates an instance of defaultL1.
     */
    public DefaultL1() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}