/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.l3vpnsvcext.rev20160730.l3vpnsvcext.classprofile;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of qosClass.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultQosClass extends InnerModelObject 
        implements QosClass, MultiInstanceObject<QosClassKeys> {
    protected String classId;
    protected short rateLimit;
    protected short priorityLevel;
    protected short guaranteedBwPercent;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public String classId() {
        return classId;
    }

    @Override
    public short rateLimit() {
        return rateLimit;
    }

    @Override
    public short priorityLevel() {
        return priorityLevel;
    }

    @Override
    public short guaranteedBwPercent() {
        return guaranteedBwPercent;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void classId(String classId) {
        valueLeafFlags.set(LeafIdentifier.CLASSID.getLeafIndex());
        this.classId = classId;
    }

    @Override
    public void rateLimit(short rateLimit) {
        valueLeafFlags.set(LeafIdentifier.RATELIMIT.getLeafIndex());
        this.rateLimit = rateLimit;
    }

    @Override
    public void priorityLevel(short priorityLevel) {
        valueLeafFlags.set(LeafIdentifier.PRIORITYLEVEL.getLeafIndex());
        this.priorityLevel = priorityLevel;
    }

    @Override
    public void guaranteedBwPercent(short guaranteedBwPercent) {
        valueLeafFlags.set(LeafIdentifier.GUARANTEEDBWPERCENT.getLeafIndex());
        this.guaranteedBwPercent = guaranteedBwPercent;
    }

    @Override
    public int hashCode() {
        return Objects.hash(classId, rateLimit, priorityLevel, guaranteedBwPercent, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultQosClass) {
            DefaultQosClass other = (DefaultQosClass) obj;
            return
                Objects.equals(classId, other.classId) &&
                Objects.equals(rateLimit, other.rateLimit) &&
                Objects.equals(priorityLevel, other.priorityLevel) &&
                Objects.equals(guaranteedBwPercent, other.guaranteedBwPercent) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("classId=" + classId)
            .add("rateLimit=" + rateLimit)
            .add("priorityLevel=" + priorityLevel)
            .add("guaranteedBwPercent=" + guaranteedBwPercent)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultQosClass.
     */
    public DefaultQosClass() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}