/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.sitesecurityencryption;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.sitesecurityencryption.encryption.EncryptionProfile;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.sitesecurityencryption.encryption.LayerEnum;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of encryption.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultEncryption extends InnerModelObject implements Encryption {
    protected boolean enabled;
    protected LayerEnum layer;
    protected BitSet valueLeafFlags = new BitSet();
    protected EncryptionProfile encryptionProfile;

    @Override
    public boolean enabled() {
        return enabled;
    }

    @Override
    public LayerEnum layer() {
        return layer;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public EncryptionProfile encryptionProfile() {
        return encryptionProfile;
    }

    @Override
    public void enabled(boolean enabled) {
        valueLeafFlags.set(LeafIdentifier.ENABLED.getLeafIndex());
        this.enabled = enabled;
    }

    @Override
    public void layer(LayerEnum layer) {
        valueLeafFlags.set(LeafIdentifier.LAYER.getLeafIndex());
        this.layer = layer;
    }

    @Override
    public void encryptionProfile(EncryptionProfile encryptionProfile) {
        this.encryptionProfile = encryptionProfile;
    }

    @Override
    public int hashCode() {
        return Objects.hash(enabled, layer, valueLeafFlags, encryptionProfile);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultEncryption) {
            DefaultEncryption other = (DefaultEncryption) obj;
            return
                Objects.equals(enabled, other.enabled) &&
                Objects.equals(layer, other.layer) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.equals(encryptionProfile, other.encryptionProfile);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("enabled=" + enabled)
            .add("layer=" + layer)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("encryptionProfile=" + encryptionProfile)
            .toString();
    }


    /**
     * Creates an instance of defaultEncryption.
     */
    public DefaultEncryption() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}