/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.flowdefinition;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.flowdefinition.matchflow.ProtocolFieldUnion;
import org.onosproject.yang.gen.v1.yrtietfinettypes.rev20130715.yrtietfinettypes.Ipv4Prefix;
import org.onosproject.yang.gen.v1.yrtietfinettypes.rev20130715.yrtietfinettypes.Ipv6Prefix;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of matchFlow.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultMatchFlow extends InnerModelObject implements MatchFlow {
    protected short dscp;
    protected short tos;
    protected short dot1p;
    protected Ipv4Prefix ipv4SrcPrefix;
    protected Ipv6Prefix ipv6SrcPrefix;
    protected Ipv4Prefix ipv4DstPrefix;
    protected Ipv6Prefix ipv6DstPrefix;
    protected int l4SrcPort;
    protected int l4DstPort;
    protected ProtocolFieldUnion protocolField;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public short dscp() {
        return dscp;
    }

    @Override
    public short tos() {
        return tos;
    }

    @Override
    public short dot1p() {
        return dot1p;
    }

    @Override
    public Ipv4Prefix ipv4SrcPrefix() {
        return ipv4SrcPrefix;
    }

    @Override
    public Ipv6Prefix ipv6SrcPrefix() {
        return ipv6SrcPrefix;
    }

    @Override
    public Ipv4Prefix ipv4DstPrefix() {
        return ipv4DstPrefix;
    }

    @Override
    public Ipv6Prefix ipv6DstPrefix() {
        return ipv6DstPrefix;
    }

    @Override
    public int l4SrcPort() {
        return l4SrcPort;
    }

    @Override
    public int l4DstPort() {
        return l4DstPort;
    }

    @Override
    public ProtocolFieldUnion protocolField() {
        return protocolField;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void dscp(short dscp) {
        valueLeafFlags.set(LeafIdentifier.DSCP.getLeafIndex());
        this.dscp = dscp;
    }

    @Override
    public void tos(short tos) {
        valueLeafFlags.set(LeafIdentifier.TOS.getLeafIndex());
        this.tos = tos;
    }

    @Override
    public void dot1p(short dot1p) {
        valueLeafFlags.set(LeafIdentifier.DOT1P.getLeafIndex());
        this.dot1p = dot1p;
    }

    @Override
    public void ipv4SrcPrefix(Ipv4Prefix ipv4SrcPrefix) {
        valueLeafFlags.set(LeafIdentifier.IPV4SRCPREFIX.getLeafIndex());
        this.ipv4SrcPrefix = ipv4SrcPrefix;
    }

    @Override
    public void ipv6SrcPrefix(Ipv6Prefix ipv6SrcPrefix) {
        valueLeafFlags.set(LeafIdentifier.IPV6SRCPREFIX.getLeafIndex());
        this.ipv6SrcPrefix = ipv6SrcPrefix;
    }

    @Override
    public void ipv4DstPrefix(Ipv4Prefix ipv4DstPrefix) {
        valueLeafFlags.set(LeafIdentifier.IPV4DSTPREFIX.getLeafIndex());
        this.ipv4DstPrefix = ipv4DstPrefix;
    }

    @Override
    public void ipv6DstPrefix(Ipv6Prefix ipv6DstPrefix) {
        valueLeafFlags.set(LeafIdentifier.IPV6DSTPREFIX.getLeafIndex());
        this.ipv6DstPrefix = ipv6DstPrefix;
    }

    @Override
    public void l4SrcPort(int l4SrcPort) {
        valueLeafFlags.set(LeafIdentifier.L4SRCPORT.getLeafIndex());
        this.l4SrcPort = l4SrcPort;
    }

    @Override
    public void l4DstPort(int l4DstPort) {
        valueLeafFlags.set(LeafIdentifier.L4DSTPORT.getLeafIndex());
        this.l4DstPort = l4DstPort;
    }

    @Override
    public void protocolField(ProtocolFieldUnion protocolField) {
        valueLeafFlags.set(LeafIdentifier.PROTOCOLFIELD.getLeafIndex());
        this.protocolField = protocolField;
    }

    @Override
    public int hashCode() {
        return Objects.hash(dscp, tos, dot1p, ipv4SrcPrefix, ipv6SrcPrefix, ipv4DstPrefix, ipv6DstPrefix, l4SrcPort, l4DstPort, protocolField, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultMatchFlow) {
            DefaultMatchFlow other = (DefaultMatchFlow) obj;
            return
                Objects.equals(dscp, other.dscp) &&
                Objects.equals(tos, other.tos) &&
                Objects.equals(dot1p, other.dot1p) &&
                Objects.equals(ipv4SrcPrefix, other.ipv4SrcPrefix) &&
                Objects.equals(ipv6SrcPrefix, other.ipv6SrcPrefix) &&
                Objects.equals(ipv4DstPrefix, other.ipv4DstPrefix) &&
                Objects.equals(ipv6DstPrefix, other.ipv6DstPrefix) &&
                Objects.equals(l4SrcPort, other.l4SrcPort) &&
                Objects.equals(l4DstPort, other.l4DstPort) &&
                Objects.equals(protocolField, other.protocolField) &&
                Objects.equals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("dscp=" + dscp)
            .add("tos=" + tos)
            .add("dot1p=" + dot1p)
            .add("ipv4SrcPrefix=" + ipv4SrcPrefix)
            .add("ipv6SrcPrefix=" + ipv6SrcPrefix)
            .add("ipv4DstPrefix=" + ipv4DstPrefix)
            .add("ipv6DstPrefix=" + ipv6DstPrefix)
            .add("l4SrcPort=" + l4SrcPort)
            .add("l4DstPort=" + l4DstPort)
            .add("protocolField=" + protocolField)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultMatchFlow.
     */
    public DefaultMatchFlow() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}