/*
 * Decompiled with CFR 0.152.
 */
package org.onosproject.common.app;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.io.ByteStreams;
import com.google.common.io.Files;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.HierarchicalConfiguration;
import org.apache.commons.configuration.XMLConfiguration;
import org.apache.commons.lang.StringUtils;
import org.onlab.util.Tools;
import org.onosproject.app.ApplicationDescription;
import org.onosproject.app.ApplicationEvent;
import org.onosproject.app.ApplicationException;
import org.onosproject.app.ApplicationStoreDelegate;
import org.onosproject.app.DefaultApplicationDescription;
import org.onosproject.core.ApplicationRole;
import org.onosproject.core.Version;
import org.onosproject.security.AppPermission;
import org.onosproject.security.Permission;
import org.onosproject.store.AbstractStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ApplicationArchive
extends AbstractStore<ApplicationEvent, ApplicationStoreDelegate> {
    private static Logger log = LoggerFactory.getLogger(ApplicationArchive.class);
    private static final String XML_MAGIC = "<?xml ";
    private static final String ZIP_MAGIC = "PK";
    private static final String APP_MAGIC = "<app ";
    private static final int APP_MAGIC_DEPTH = 1024;
    private static final String NAME = "[@name]";
    private static final String ORIGIN = "[@origin]";
    private static final String VERSION = "[@version]";
    private static final String FEATURES_REPO = "[@featuresRepo]";
    private static final String FEATURES = "[@features]";
    private static final String APPS = "[@apps]";
    private static final String DESCRIPTION = "description";
    private static final String UTILITY = "utility";
    private static final String CATEGORY = "[@category]";
    private static final String URL = "[@url]";
    private static final String TITLE = "[@title]";
    private static final String ROLE = "security.role";
    private static final String APP_PERMISSIONS = "security.permissions.app-perm";
    private static final String NET_PERMISSIONS = "security.permissions.net-perm";
    private static final String JAVA_PERMISSIONS = "security.permissions.java-perm";
    private static final String JAR = ".jar";
    private static final String OAR = ".oar";
    private static final String APP_XML = "app.xml";
    private static final String APP_PNG = "app.png";
    private static final String M2_PREFIX = "m2";
    private static final String FEATURES_XML = "features.xml";
    private static final String ROOT = "../";
    private static final String M2_ROOT = "system/";
    private static final String APPS_ROOT = "apps/";
    private File root = new File("../");
    private File appsDir = new File(this.root, "apps/");
    private File m2Dir = new File("system/");

    protected void setRootPath(String root) {
        this.root = new File(root);
        this.appsDir = new File(this.root, APPS_ROOT);
        this.m2Dir = new File(M2_ROOT);
    }

    public String getRootPath() {
        return this.root.getPath();
    }

    public Set<String> getApplicationNames() {
        ImmutableSet.Builder names = ImmutableSet.builder();
        File[] files = this.appsDir.listFiles(File::isDirectory);
        if (files != null) {
            for (File file : files) {
                names.add((Object)file.getName());
            }
        }
        return names.build();
    }

    public long getUpdateTime(String appName) {
        return this.appFile(appName, APP_XML).lastModified();
    }

    public ApplicationDescription getApplicationDescription(String appName) {
        try {
            XMLConfiguration cfg = new XMLConfiguration();
            cfg.setAttributeSplittingDisabled(true);
            cfg.setDelimiterParsingDisabled(true);
            cfg.load(this.appFile(appName, APP_XML));
            return this.loadAppDescription(cfg);
        }
        catch (Exception e) {
            throw new ApplicationException("Unable to get app description", (Throwable)e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public synchronized ApplicationDescription saveApplication(InputStream stream) {
        try (InputStream ais = stream;){
            byte[] cache = ByteStreams.toByteArray((InputStream)ais);
            ByteArrayInputStream bis = new ByteArrayInputStream(cache);
            boolean plainXml = this.isPlainXml(cache);
            ApplicationDescription desc = plainXml ? this.parsePlainAppDescription(bis) : this.parseZippedAppDescription(bis);
            Preconditions.checkState((!this.appFile(desc.name(), APP_XML).exists() ? 1 : 0) != 0, (String)"Application %s already installed", (Object)desc.name());
            boolean isSelfContainedJar = false;
            if (plainXml) {
                this.expandPlainApplication(cache, desc);
            } else {
                ((InputStream)bis).reset();
                isSelfContainedJar = this.expandZippedApplication(bis, desc);
                if (isSelfContainedJar) {
                    ((InputStream)bis).reset();
                    this.stageSelfContainedJar(bis, desc);
                }
                ((InputStream)bis).reset();
                this.saveApplication(bis, desc, isSelfContainedJar);
            }
            this.installArtifacts(desc);
            ApplicationDescription applicationDescription = desc;
            return applicationDescription;
        }
        catch (IOException e) {
            throw new ApplicationException("Unable to save application", (Throwable)e);
        }
    }

    private boolean isPlainXml(byte[] bytes) {
        return !this.substring(bytes, ZIP_MAGIC.length()).equals(ZIP_MAGIC) && (this.substring(bytes, XML_MAGIC.length()).equals(XML_MAGIC) || this.substring(bytes, 1024).contains(APP_MAGIC));
    }

    private String substring(byte[] bytes, int length) {
        return new String(bytes, 0, Math.min(bytes.length, length), StandardCharsets.UTF_8);
    }

    public synchronized void purgeApplication(String appName) {
        File appDir = new File(this.appsDir, appName);
        try {
            Tools.removeDirectory((File)appDir);
        }
        catch (IOException e) {
            throw new ApplicationException("Unable to purge application " + appName, (Throwable)e);
        }
        if (appDir.exists()) {
            throw new ApplicationException("Unable to purge application " + appName);
        }
    }

    public synchronized InputStream getApplicationInputStream(String appName) {
        try {
            File appFile = this.appFile(appName, appName + OAR);
            return new FileInputStream(appFile.exists() ? appFile : this.appFile(appName, APP_XML));
        }
        catch (FileNotFoundException e) {
            throw new ApplicationException("Application " + appName + " not found");
        }
    }

    private ApplicationDescription parseZippedAppDescription(InputStream stream) throws IOException {
        try (ZipInputStream zis = new ZipInputStream(stream);){
            ZipEntry entry;
            while ((entry = zis.getNextEntry()) != null) {
                if (entry.getName().equals(APP_XML)) {
                    byte[] data = ByteStreams.toByteArray((InputStream)zis);
                    ApplicationDescription applicationDescription = this.parsePlainAppDescription(new ByteArrayInputStream(data));
                    return applicationDescription;
                }
                zis.closeEntry();
            }
        }
        throw new IOException("Unable to locate app.xml");
    }

    private ApplicationDescription parsePlainAppDescription(InputStream stream) throws IOException {
        XMLConfiguration cfg = new XMLConfiguration();
        cfg.setAttributeSplittingDisabled(true);
        cfg.setDelimiterParsingDisabled(true);
        try {
            cfg.load(stream);
            return this.loadAppDescription(cfg);
        }
        catch (ConfigurationException e) {
            throw new IOException("Unable to parse app.xml", e);
        }
    }

    private ApplicationDescription loadAppDescription(XMLConfiguration cfg) {
        String name = cfg.getString(NAME);
        Version version = Version.version((String)cfg.getString(VERSION));
        String origin = cfg.getString(ORIGIN);
        String title = cfg.getString(TITLE);
        title = title == null ? name : title;
        String category = cfg.getString(CATEGORY, UTILITY);
        String url = cfg.getString(URL);
        byte[] icon = this.getApplicationIcon(name);
        ApplicationRole role = this.getRole(cfg.getString(ROLE));
        ImmutableSet<Permission> perms = this.getPermissions(cfg);
        String featRepo = cfg.getString(FEATURES_REPO);
        URI featuresRepo = featRepo != null ? URI.create(featRepo) : null;
        ImmutableList features = ImmutableList.copyOf((Object[])cfg.getString(FEATURES).split(","));
        String apps = cfg.getString(APPS, "");
        ImmutableList requiredApps = apps.isEmpty() ? ImmutableList.of() : ImmutableList.copyOf((Object[])apps.split(","));
        String readme = cfg.getString(DESCRIPTION);
        String desc = this.compactDescription(readme);
        return new DefaultApplicationDescription(name, version, title, desc, origin, category, url, readme, icon, role, perms, featuresRepo, (List)features, (List)requiredApps);
    }

    private boolean expandZippedApplication(InputStream stream, ApplicationDescription desc) throws IOException {
        ZipEntry entry;
        boolean isSelfContained = false;
        ZipInputStream zis = new ZipInputStream(stream);
        File appDir = new File(this.appsDir, desc.name());
        while ((entry = zis.getNextEntry()) != null) {
            if (entry.isDirectory()) continue;
            byte[] data = ByteStreams.toByteArray((InputStream)zis);
            zis.closeEntry();
            File file = new File(appDir, entry.getName());
            if (this.isTopLevel(file)) {
                Files.createParentDirs((File)file);
                Files.write((byte[])data, (File)file);
                continue;
            }
            isSelfContained = true;
        }
        zis.close();
        return isSelfContained;
    }

    private boolean isTopLevel(File file) {
        String name = file.getName();
        return name.equals(APP_XML) || name.endsWith(FEATURES_XML) || name.endsWith(JAR) || file.isDirectory();
    }

    private void stageSelfContainedJar(InputStream stream, ApplicationDescription desc) throws IOException {
        String coords = this.getSelfContainedBundleCoordinates(desc);
        if (coords == null) {
            return;
        }
        String[] f = coords.substring(4).split("/");
        String base = "m2/" + f[0].replace('.', '/') + "/" + f[1] + "/" + f[2] + "/" + f[1] + "-" + f[2];
        String jarName = base + (f.length < 4 ? "" : "-" + f[3]) + JAR;
        String featuresName = base + "-features.xml";
        File jar = this.appFile(desc.name(), jarName);
        boolean ok = jar.getParentFile().mkdirs();
        if (ok) {
            Files.write((byte[])ByteStreams.toByteArray((InputStream)stream), (File)jar);
            Files.copy((File)this.appFile(desc.name(), FEATURES_XML), (File)this.appFile(desc.name(), featuresName));
            if (!this.appFile(desc.name(), FEATURES_XML).delete()) {
                log.warn("Unable to delete self-contained application {} features.xml", (Object)desc.name());
            }
        } else {
            throw new IOException("Unable to save self-contained application " + desc.name());
        }
    }

    private String getSelfContainedBundleCoordinates(ApplicationDescription desc) {
        try {
            XMLConfiguration cfg = new XMLConfiguration();
            cfg.setAttributeSplittingDisabled(true);
            cfg.setDelimiterParsingDisabled(true);
            cfg.load(this.appFile(desc.name(), FEATURES_XML));
            return cfg.getString("feature.bundle");
        }
        catch (ConfigurationException e) {
            log.warn("Self-contained application {} has no features.xml", (Object)desc.name());
            return null;
        }
    }

    private void expandPlainApplication(byte[] stream, ApplicationDescription desc) throws IOException {
        File file = this.appFile(desc.name(), APP_XML);
        Preconditions.checkState((!file.getParentFile().exists() ? 1 : 0) != 0, (Object)"Application already installed");
        Files.createParentDirs((File)file);
        Files.write((byte[])stream, (File)file);
    }

    private void saveApplication(InputStream stream, ApplicationDescription desc, boolean isSelfContainedJar) throws IOException {
        String name = desc.name() + (isSelfContainedJar ? JAR : OAR);
        Files.write((byte[])ByteStreams.toByteArray((InputStream)stream), (File)this.appFile(desc.name(), name));
    }

    private void installArtifacts(ApplicationDescription desc) throws IOException {
        try {
            Tools.copyDirectory((File)this.appFile(desc.name(), M2_PREFIX), (File)this.m2Dir);
        }
        catch (NoSuchFileException e) {
            log.debug("Application {} has no M2 artifacts", (Object)desc.name());
        }
    }

    protected boolean setActive(String appName) {
        try {
            File active = this.appFile(appName, "active");
            Files.createParentDirs((File)active);
            return active.createNewFile() && this.updateTime(appName);
        }
        catch (IOException e) {
            log.warn("Unable to mark app {} as active", (Object)appName, (Object)e);
            throw new ApplicationException("Unable to mark app as active", (Throwable)e);
        }
    }

    protected boolean clearActive(String appName) {
        return this.appFile(appName, "active").delete() && this.updateTime(appName);
    }

    protected boolean updateTime(String appName) {
        return this.appFile(appName, APP_XML).setLastModified(System.currentTimeMillis());
    }

    protected boolean isActive(String appName) {
        return this.appFile(appName, "active").exists();
    }

    private File appFile(String appName, String fileName) {
        return new File(new File(this.appsDir, appName), fileName);
    }

    private File iconFile(String appName, String fileName) {
        return new File(new File(this.appsDir, appName), fileName);
    }

    private ImmutableSet<Permission> getPermissions(XMLConfiguration cfg) {
        ArrayList permissionList = Lists.newArrayList();
        for (Object o : cfg.getList(APP_PERMISSIONS)) {
            String name = (String)o;
            permissionList.add(new Permission(AppPermission.class.getName(), name));
        }
        Iterator iterator = cfg.getList(NET_PERMISSIONS).iterator();
        if (iterator.hasNext()) {
            Object o;
            o = iterator.next();
        }
        List fields = cfg.configurationsAt(JAVA_PERMISSIONS);
        for (HierarchicalConfiguration sub : fields) {
            String classname = sub.getString("classname");
            String name = sub.getString("name");
            String actions = sub.getString("actions");
            if (classname == null || name == null) continue;
            permissionList.add(new Permission(classname, name, actions));
        }
        return ImmutableSet.copyOf((Collection)permissionList);
    }

    private byte[] getApplicationIcon(String appName) {
        File iconFile = this.iconFile(appName, APP_PNG);
        try {
            InputStream iconStream = iconFile.exists() ? new FileInputStream(iconFile) : ApplicationArchive.class.getResourceAsStream("/app.png");
            return ByteStreams.toByteArray((InputStream)iconStream);
        }
        catch (IOException e) {
            log.warn("Unable to read app icon for app {}", (Object)appName, (Object)e);
            return new byte[0];
        }
    }

    public ApplicationRole getRole(String value) {
        if (value == null) {
            return ApplicationRole.UNSPECIFIED;
        }
        try {
            return ApplicationRole.valueOf((String)value.toUpperCase(Locale.ENGLISH));
        }
        catch (IllegalArgumentException e) {
            log.debug("Unknown role value: %s", (Object)value);
            return ApplicationRole.UNSPECIFIED;
        }
    }

    private String compactDescription(String sentence) {
        if (StringUtils.isNotEmpty((String)sentence)) {
            if (StringUtils.contains((String)sentence, (String)".")) {
                return StringUtils.substringBefore((String)sentence, (String)".") + ".";
            }
            return sentence;
        }
        return sentence;
    }
}

