package net.officefloor.tutorial.jwtauthorityhttpserver;

import java.util.concurrent.ThreadLocalRandom;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import net.officefloor.server.http.HttpException;
import net.officefloor.server.http.HttpStatus;
import net.officefloor.web.HttpObject;
import net.officefloor.web.ObjectResponse;
import net.officefloor.web.jwt.authority.AccessToken;
import net.officefloor.web.jwt.authority.JwtAuthority;
import net.officefloor.web.jwt.authority.RefreshToken;

/**
 * Undertakes login.
 * 
 * @author Daniel Sagenschneider
 */
public class JwtTokens {

	// START SNIPPET: login
	@Data
	@HttpObject
	@RequiredArgsConstructor
	@AllArgsConstructor
	public static class Credentials {
		private String username;
		private String password;
	}

	@Data
	@RequiredArgsConstructor
	@AllArgsConstructor
	public static class Tokens {
		private String refreshToken;
		private String accessToken;
	}

	public void login(Credentials credentials, JwtAuthority<Identity> authority, ObjectResponse<Tokens> response) {

		// Mock authentication
		// (production solution would check appropriate user store)
		// (or use potential OpenId third party login)
		if ((credentials.getUsername() == null) || (!credentials.getUsername().equals(credentials.getPassword()))) {
			throw new HttpException(HttpStatus.UNAUTHORIZED);
		}

		// Create the identity and claims
		Identity identity = new Identity(credentials.username);
		Claims claims = createClaims(credentials.username);

		// Create the refresh and access tokens
		RefreshToken refreshToken = authority.createRefreshToken(identity);
		AccessToken accessToken = authority.createAccessToken(claims);

		// Send response
		response.send(new Tokens(refreshToken.getToken(), accessToken.getToken()));
	}

	private static Claims createClaims(String username) {

		// Mock claims
		// (claim information should be pulled from user store)
		String[] roles = new String[] { "tutorial" };

		// Provide random value (so access tokens are different)
		// (not necessary but due to speed of tests, gets same access token)
		int randomValue = ThreadLocalRandom.current().nextInt();

		// Return the claims
		return new Claims(username, randomValue, roles);
	}
	// END SNIPPET: login

	// START SNIPPET: refresh
	@Data
	@HttpObject
	@AllArgsConstructor
	@RequiredArgsConstructor
	public static class Token {
		private String token;
	}

	public void refreshAccessToken(Token request, JwtAuthority<Identity> authority, ObjectResponse<Token> response) {

		// Obtain the identity from refresh token
		Identity identity = authority.decodeRefreshToken(request.token);

		// Create a new access token
		Claims claims = createClaims(identity.getId());
		AccessToken accessToken = authority.createAccessToken(claims);

		// Send refreshed access token
		response.send(new Token(accessToken.getToken()));
	}
	// END SNIPPET: refresh

}