/*
 * This file is part of ScriptController, licensed under the MIT License.
 *
 *  Copyright (c) lucko (Luck) <luck@lucko.me>
 *  Copyright (c) contributors
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in all
 *  copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 *  SOFTWARE.
 */

package me.lucko.scriptcontroller.environment.loader;

import me.lucko.scriptcontroller.environment.ScriptEnvironment;

import java.util.Arrays;
import java.util.Collection;

/**
 * An object capable of loadings scripts and monitoring them for updates.
 */
public interface ScriptLoader extends AutoCloseable {

    /**
     * Gets the script environment this loader is operating in
     *
     * @return the parent environment
     */
    ScriptEnvironment getEnvironment();

    /**
     * Loads and watches a script
     *
     * @param paths the path to watch
     */
    default void watch(String... paths) {
        this.watchAll(Arrays.asList(paths));
    }

    /**
     * Loads and watches a collection of scripts
     *
     * @param paths the paths to watch
     */
    void watchAll(Collection<String> paths);

    /**
     * Unloads a script
     *
     * @param paths the path to unwatch
     */
    default void unwatch(String... paths) {
        this.unwatchAll(Arrays.asList(paths));
    }

    /**
     * Unloads a collection of scripts
     *
     * @param paths the paths to unwatch
     */
    void unwatchAll(Collection<String> paths);

    @Override
    void close() throws Exception;
}
