package com.openfin.desktop.win32;

import com.sun.jna.platform.win32.Advapi32Util;
import com.sun.jna.platform.win32.WinReg;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;

public class RegistryHelper {
    private final static Logger logger = LoggerFactory.getLogger(RegistryHelper.class.getName());

    // RVM Key Precedence
    // HKEY_CURRENT_USER > HKEY_LOCAL_MACHINE > Group Policy Registry > Registry
    private final static String RVM_INSTALL_DIR_REG_KEY = "Software\\OpenFin\\RVM\\Settings\\Deployment";  // Registry key
    private final static String RVM_INSTALL_DIR_GP_KEY = "Software\\Microsoft\\Windows\\CurrentVersion\\Policies\\OpenFin\\RVM\\Settings\\Deployment";  // Group Policy key
    private final static String RVM_INSTALL_DIR_REG_VALUE_NAME = "rvmInstallDirectory";
    private static final String WIN_LOCAL_APP_DATA = "LOCALAPPDATA";
    private static final String RUNTIME_INSTALL_DIRECTORY = "runtimeDirectory"; // Runtime binaries will be stored here
    private static final String RUNTIME_CACHE_DIRECTORY = "runtimeCache";   //

    private static String registryUserGetStringValue(String key, String valueName) {
        String value = null;
        try {
            logger.debug(String.format("Reading HKCU registry %s %s", key, valueName));
            value = Advapi32Util.registryGetStringValue(WinReg.HKEY_CURRENT_USER, key, valueName);
        } catch (Exception ex) {
            // thrown if the KEY does not exist, not an error
            logger.debug(String.format("Error reading registry %s %s", key, valueName));
        }
        return value;
    }

    private static String registryMachineGetStringValue(String key, String valueName) {
        String value = null;
        try {
            logger.debug(String.format("Reading HKLM registry %s %s", key, valueName));
            value = Advapi32Util.registryGetStringValue(WinReg.HKEY_LOCAL_MACHINE, key, valueName);
        } catch (Exception ex) {
            // thrown if the KEY does not exist, not an error
            logger.debug(String.format("Error reading registry %s %s", key, valueName));
        }
        return value;
    }

    private static String getRVMStringValue(String valueName) {
        String value = registryUserGetStringValue(RVM_INSTALL_DIR_GP_KEY, valueName);
        if (value == null) {
            value = registryMachineGetStringValue(RVM_INSTALL_DIR_GP_KEY, valueName);
        }
        if (value == null) {
            value = registryUserGetStringValue(RVM_INSTALL_DIR_REG_KEY, valueName);
        }
        if (value == null) {
            value = registryMachineGetStringValue(RVM_INSTALL_DIR_REG_KEY, valueName);
        }
        return value;
    }

    private static String getOpenFinDirectory(String regKey, String extraPathForDefault) {
        String value = getRVMStringValue(regKey);
        if (value == null) {
            logger.debug(String.format("%s missing from Registry.  Getting default ", regKey));
            String localAppData = (java.lang.System.getenv(WIN_LOCAL_APP_DATA));
            if (localAppData != null) {
                value = localAppData + File.separator + "OpenFin";
                if (extraPathForDefault != null) {
                    value = value + File.separator + extraPathForDefault;
                }
            } else {
                logger.debug(String.format("env variable %s not set ", WIN_LOCAL_APP_DATA));
            }
        }
        return value;
    }

    public static String getRVMInstallDirectory() {
        return getOpenFinDirectory(RVM_INSTALL_DIR_REG_VALUE_NAME, null);
    }

    public static String getRuntimeInstallDirectory() {
        return getOpenFinDirectory(RUNTIME_INSTALL_DIRECTORY, "runtime");
    }

    public static void main(String[] argv) {
        try {

            java.lang.System.out.println(registryUserGetStringValue(RVM_INSTALL_DIR_REG_KEY, RVM_INSTALL_DIR_REG_VALUE_NAME));

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
