package com.openfin.desktop.channel;

import com.openfin.desktop.AckListener;
import com.openfin.desktop.AsyncCallback;
import com.openfin.desktop.DesktopConnection;
import com.openfin.desktop.JsonUtils;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class NotificationClient {
    private final static Logger logger = LoggerFactory.getLogger(NotificationClient.class.getName());

    // Actions supported by Notfication service
    private static String ClearAppNotifications = "clear-app-notifications";
    private static String ClearNotifications    = "clear-notification";
    private static String CreateNotification    = "create-notification";
    private static String GetAppNotifications   = "fetch-app-notifications";
    private static String ToggleNotificationCenter = "toggle-notification-center";
    private static String ServiceChannelName    = "of-notifications-service-v1";
//    private static String ServiceManifestUrl    = "https://cdn.openfin.co/services/openfin/notifications/app.json";
    // Events from Notification service
    private static String NotificationClicked       = "notification-clicked";
    private static String NotificationButtonClicked = "notification-button-clicked";
    private static String NotificationClosed        = "notification-closed";

    private static String ButtonIndex = "buttonIndex";

    private ChannelClient channelClient;

    public NotificationClient(DesktopConnection desktopConnection, AckListener listener) {
        desktopConnection.getChannel(ServiceChannelName).connect(
                new AsyncCallback<ChannelClient>() {
                    @Override
                    public void onSuccess(ChannelClient client) {
                        NotificationClient.this.channelClient = client;
                        listener.onSuccess(null);
                        logger.debug(String.format("Connected to Notification service %s", ServiceChannelName));
                    }
                });
    }

    public void create(NotificationOptions options, AckListener listener) {
        this.channelClient.dispatch(CreateNotification, options.getJson(), listener);
    }

    public void clear(String id, AckListener listener) {
        NotificationOptions options = new NotificationOptions();
        options.setId(id);
        this.channelClient.dispatch(ClearNotifications, options.getJson(), listener);
    }

    public void clearAll(AckListener listener) {
        this.channelClient.dispatch(ClearAppNotifications, null, listener);
    }

    public void toggleNotificationCenter(AckListener listener) {
        this.channelClient.dispatch(ToggleNotificationCenter, null, listener);
    }

    public void addNotificationListener(NotificationListener listener) {
        this.channelClient.register("event", new ChannelAction() {
            @Override
            public JSONObject invoke(String action, JSONObject payload, JSONObject senderIdentity) {
                logger.debug(String.format("Notification event %s", payload.toString()));
                String type = payload.getString("type");
                JSONObject notification = payload.getJSONObject("notification");
                if (NotificationClicked.equals(type)) {
                    if (listener != null) {
                        listener.onClick(NotificationOptions.fromJson(notification));
                    }
                }
                else if (NotificationButtonClicked.equals(type)) {
                    if (listener != null) {
                        NotificationOptions options = NotificationOptions.fromJson(notification);
                        options.setButtonIndex(JsonUtils.getIntegerValue(payload, ButtonIndex, -1));
                        listener.onButtonClick(options);
                    }
                }
                else if (NotificationClosed.equals(type)) {
                    if (listener != null) {
                        listener.onClose(NotificationOptions.fromJson(notification));
                    }
                }
                return null;
            }
        });
    }
}
