package com.openfin.desktop.channel;

import com.openfin.desktop.*;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LayoutClient {

    private final static Logger logger = LoggerFactory.getLogger(LayoutClient.class.getName());

    private static String ServiceChannelName    = "of-layouts-service-v1";
    // Actions supported by Notfication service
    private static String GENERATE_WORKSPACE = "GENERATE-WORKSPACE";
    private static String RESTORE_WORKSPACE = "RESTORE-WORKSPACE";

    // Actions supported by Snap&Dock
    private static String UNDOCK_WINDOW = "UNDOCK-WINDOW";

    // Actions for tabbing
    private static String  TAB_WINDOW_TO_WINDOW = "TAB_WINDOW_TO_WINDOW";
    private static String  REMOVETAB = "REMOVETAB";

    private ChannelClient channelClient;

    public LayoutClient(DesktopConnection desktopConnection, AckListener listener) {
        desktopConnection.getChannel(ServiceChannelName).connect(
                new AsyncCallback<ChannelClient>() {
                    @Override
                    public void onSuccess(ChannelClient client) {
                        LayoutClient.this.channelClient = client;
                        Ack ack = new Ack(null, LayoutClient.this);
                        listener.onSuccess(ack);
                        logger.debug(String.format("Connected to Layout service %s", ServiceChannelName));
                    }
                });
    }

    public void undockWindow(String applicationUuid, String windowName, AckListener listener) {
        JSONObject payload = new JSONObject();
        payload.put("uuid", applicationUuid);
        payload.put("name", windowName);
        this.channelClient.dispatch(UNDOCK_WINDOW, payload, listener);
    }

    /**
     * Add a window, with identity, as a tab to the context the target window belongs to
     *
     * @param target target window
     * @param identity window to add
     * @param listener AckListener for this request
     */
    public void tabWindows(WindowIdentity target, WindowIdentity identity, AckListener listener) {
        JSONObject payload = new JSONObject();
        payload.put("targetWindow", target.toJsonObject());
        payload.put("windowToAdd", identity.toJsonObject());
        this.channelClient.dispatch(TAB_WINDOW_TO_WINDOW, payload, listener);
    }

    public void removeTab(WindowIdentity identity, AckListener listener) {
        JSONObject payload = identity.toJsonObject();
        this.channelClient.dispatch(REMOVETAB, payload, listener);
    }

    public void generateWorkspace(AsyncCallback<JSONObject> callback, AckListener listener) {
        this.channelClient.dispatch(GENERATE_WORKSPACE, null, new AckListener() {
            @Override
            public void onSuccess(Ack ack) {
                JSONObject ackObject = ack.getJsonObject();
                if (ackObject.has("data")) {
                    JSONObject data = ackObject.getJSONObject("data");
                    if (data.has("result")) {
                        JSONObject result = data.getJSONObject("result");
                        callback.onSuccess(result);
                    } else {
                        logger.debug(String.format("result element missing from %s", ackObject.toString()));
                        listener.onError(ack);
                    }
                } else {
                    logger.debug(String.format("data element missing from %s", ackObject.toString()));
                    listener.onError(ack);
                }
                logger.info(String.format("workspace %s", ack.getJsonObject().toString()));
            }
            @Override
            public void onError(Ack ack) {
                listener.onError(ack);
            }
        });
    }

    public void retoreWorkspace(JSONObject workspace, AckListener listener) {
        this.channelClient.dispatch(RESTORE_WORKSPACE, workspace, new AckListener() {
            @Override
            public void onSuccess(Ack ack) {
                logger.info(String.format("workspace %s", ack.getJsonObject().toString()));
            }
            @Override
            public void onError(Ack ack) {
                listener.onError(ack);
            }
        });
    }

}
