package com.openfin.desktop.channel;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Objects;

import com.openfin.desktop.ActionEvent;
import com.openfin.desktop.EventListener;
import org.json.JSONObject;

import com.openfin.desktop.AckListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChannelProvider extends ChannelBase {
	private final static Logger logger = LoggerFactory.getLogger(ChannelProvider.class.getName());

	private ArrayList<EndpointIdentity> clients;

	ChannelProvider(Channel messageChannelFactory, EndpointIdentity endpointIdentity) {
		super(messageChannelFactory, endpointIdentity);
		this.clients = new ArrayList<>();
		this.channel.addChannelListener(new ChannelListener() {
			@Override
			public void onChannelConnect(ConnectionEvent connectionEvent) {
			}

			@Override
			public void onChannelDisconnect(ConnectionEvent connectionEvent) {
				processDisconnection(connectionEvent.getUuid(), connectionEvent.getName());
			}
		});

		JSONObject eventListenerPayload = new JSONObject();
		eventListenerPayload.put("topic", "channel");
		eventListenerPayload.put("type", "client-disconnected");
		this.channel.getDesktopConnection().addEventCallback(eventListenerPayload, new EventListener() {
			@Override
			public void eventReceived(ActionEvent actionEvent) {
				JSONObject event = actionEvent.getEventObject();
				if (ChannelProvider.this.channel.getName().equals(event.getString("channelName"))) {
					String uuid = event.getString("uuid");
					String name = event.getString("name");
					ChannelProvider.this.processDisconnection(uuid, name);
					ChannelProvider.this.channel.fireChannelDisconnectEvent(endpointIdentity.getChannelId(),
							uuid, name, endpointIdentity.getChannelName(), endpointIdentity.getEndpointId());
				}
			}
		}, null, this);

	}
	
	/**
	 * Destroy the channel
	 * @param ackListener AckListener for the request
	 */
	public void destroy(AckListener ackListener) {
		this.channel.destroy(this, ackListener);
	}

	public void processConnection(JSONObject clientIdentity, JSONObject connectionPayload) {
		String clientUuid = clientIdentity.has("uuid") ? clientIdentity.getString("uuid") : null;
		String clientName = clientIdentity.has("name") ? clientIdentity.getString("name") : null;
		String clientEndpointId = clientIdentity.has("endpointId") ? clientIdentity.getString("endpointId") : null;
		
		EndpointIdentity clientEndpointIdentity = new EndpointIdentity(this.getChannelName(), this.getChannelId(), clientUuid, clientName, clientEndpointId);
		this.clients.add(clientEndpointIdentity);
		this.channel.fireChannelConnectEvent(clientEndpointIdentity.getChannelId(), clientEndpointIdentity.getUuid(),
				clientEndpointIdentity.getName(), clientEndpointIdentity.getChannelName(),
				clientEndpointIdentity.getEndpointId());
	}

	private void processDisconnection(String uuid, String name) {
		logger.debug(String.format("Client disconnected %s %s from channel %s", uuid, name, this.channel.getName()));
		Iterator<EndpointIdentity> allClients = this.clients.iterator();
		while (allClients.hasNext()) {
			EndpointIdentity client = allClients.next();
			if (Objects.equals(client.getUuid(), uuid) && Objects.equals(client.getName(), name)) {
				allClients.remove();
			}
		}
	}

	/**
	 * Publish an action and payload to every connected client.
	 * @param action Name of the action to be invoked by the channel client
	 * @param actionPayload Payload to be sent along with the action.
	 * @param ackListener AckListener for the request
	 */
	public void publish(String action, JSONObject actionPayload, AckListener ackListener) {
		for (EndpointIdentity client : this.clients) {
			//the logic in core, if destinationId has channelId, it means the message is intended for channel provider
			//here create the new endpointIdentity that has no channelId
			EndpointIdentity destId = new EndpointIdentity(client.getChannelName(), null, client.getUuid(), client.getName(), client.getEndpointId());
			this.dispatch(destId.toJSON(), action, actionPayload, ackListener);
		}
	}
	
	/**
	 * Register an action to be called by ChannelClient
	 * @param action Name of the action to be invoked by the channel client
	 * @param listener Function representing the action to be taken on a client dispatch.
	 * @return if the action was registered successfully
	 */
	public boolean register(String action, ChannelAction listener) {
		return super.register(action, listener);
	}

}
