package com.openfin.desktop;

import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A class representing the options for a Window.
 */
public class WindowOptions extends JsonBean {
    private final static Logger logger = LoggerFactory.getLogger(WindowOptions.class.getName());

    private int resizeRegionBottomRightCorner = 4;
    private int resizeRegionSize = 2;

	private AcceleratorOptions acceleratorOptions;
	private LayoutOptions layoutOptions;

    /**
     * Default constructor
     */
    public WindowOptions() {
    }

    /**
     * Contructor with name and url
     *
     * @param name name of the Window
     * @param url URL of the window
     *
     */
    public WindowOptions(String name, String url) {
        this();
        setName(name);
        setUrl(url);
    }

    /**
     * Constructs an instance with a new underlying JSONObject
     * @param options underlying options
     */
    public WindowOptions(JSONObject options) {
    	super(options);
    	this.getAccelerator();
    	this.getLayoutOptions();
    }

    /**
     *
     * The ShowTaskbarIcon property represents a flag to show
     * the Window's icon in the taskbar.
     * Default: false
     *
     * @param showTaskbarIcon show taskbar icon if true
     */
    public void setShowTaskbarIcon(boolean showTaskbarIcon) {
        try {
            json.put("showTaskbarIcon", showTaskbarIcon);
        } catch (JSONException e) {
            logger.error("Error setting showTaskbarIcon", e);
        }
    }

    /**
     * Get name of the window
     *
     * @return name
     */
    public String getName() {
        return getString("name");
    }

    /**
     * Set name of the window
     *
     * @param name name of the Window
     */
    public void setName(String name) {
        try {
            json.put("name", name);
        } catch (JSONException e) {
            logger.error("Error setting name", e);
        }
    }

    /**
     * Get URL of the window
     *
     * @return url of the Window
     */
    public String getUrl() {
        return getString("url");
    }

    /**
     * Set name of the window
     *
     * @param url name of the window
     */
    public void setUrl(String url) {
        try {
            json.put("url", url);
        } catch (JSONException e) {
            logger.error("Error setting URL", e);
        }
    }

    /**
     *
     * The ShowTaskbarIcon property represents a flag to show
     * the Window's icon in the taskbar.
     *
     * Default: false
     *
     * @return value of showTaskbarIcon
     */
    public boolean getShowTaskbarIcon() {
        return getBooleanValue("showTaskbarIcon");
    }

    public void setDefaultWidth(int defaultWidth) {
        try {
            json.put("defaultWidth", defaultWidth);
        } catch (JSONException e) {
            logger.error("Error setting default width", e);
        }
    }

    /**
     *
     * The DefaultWidth property represents the window's
     * default width in pixels.
     * Default: 800
     *
     * @return value of defaultWidth
     */
    public int getDefaultWidth() {
        return getIntegerValue("defaultWidth");
    }

    /**
     *
     * The DefaultWidth property represents the window's
     * default height in pixels.
     * Default: 500
     *
     * @param defaultHeight value of defaultHeight
     */
    public void setDefaultHeight(int defaultHeight) {
        try {
            json.put("defaultHeight", defaultHeight);
        } catch (JSONException e) {
            logger.error("Error setting default height", e);
        }
    }

    /**
     *
     * The DefaultWidth property represents the window's
     * default height in pixels.
     *
     * Default: 500
     *
     * @return value of defaultHeight
     */
    public int getDefaultHeight() {
        return getIntegerValue("defaultHeight");
    }

    /**
     *
     * The DefaultTop property represents the window's
     * default top position.
     * Default: 100
     *
     * @param defaultTop value of defaultTop
     */
    public void setDefaultTop(int defaultTop) {
        try {
            json.put("defaultTop", defaultTop);
        } catch (JSONException e) {
            logger.error("Error setting top", e);
        }
    }

    /**
     *
     * The DefaultTop property represents the window's
     * default top position.
     * Default: 100
     *
     * @return value of defaultTop
     */
    public int getDefaultTop() {
        return getIntegerValue("defaultTop");
    }

    /**
     *
     * The DefaultLeft property represents the window's
     * default left position.
     * Default: 100
     *
     * @param defaultLeft value of defaultLeft
     */
    public void setDefaultLeft(int defaultLeft) {
        try {
            json.put("defaultLeft", defaultLeft);
        } catch (JSONException e) {
            logger.error("Error setting left", e);
        }
    }

    /**
     *
     * The DefaultLeft property represents the window's
     * default left position.
     * Default: 100
     *
     * @return value of defaultLeft
     */
    public int getDefaultLeft() {
        return getIntegerValue("defaultLeft");
    }

    /**
     *
     * The Frame property represents a flag to show the frame.
     * Default: true
     *
     * @param frame value of frame
     */
    public void setFrame(boolean frame) {
        try {
            json.put("frame", frame);
        } catch (JSONException e) {
            logger.error("Error setting frame", e);
        }
    }

    /**
     *
     * The Frame property represents a flag to show the frame.
     * Default: true
     *
     * @return value of frame
     */
    public boolean getFrame() {
        return getBooleanValue("frame");
    }

    /**
     *
     * The Resizable property represents a flag which allows
     * the user to resize the window.
     * This property will be deprecated in a future release.
     * Default: true
     *
     * @param resizable value of resizable
     */
    public void setResizable(boolean resizable) {
        try {
            json.put("resizable", resizable);
        } catch (JSONException e) {
            logger.error("Error setting resizable", e);
        }
    }

    /**
     *
     * The Resizable property represents a flag which allows
     * the user to resize the window.
     * This property will be deprecated in a future release.
     * Default: true
     *
     * @return value of resizable
     */
    public boolean getResizable() {
        return getBooleanValue("resizable");
    }

    /**
     *
     * The AutoShow property represents a flag to automatically
     * show the Window when it is created.
     * Default: false
     *
     * @param autoShow value of autoShow
     */
    public void setAutoShow(boolean autoShow) {
        try {
            json.put("autoShow", autoShow);
        } catch (JSONException e) {
            logger.error("Error setting autoShow", e);
        }
    }

    /**
     *
     * The AutoShow property represents a flag to automatically
     * show the Window when it is created.
     * Default: false
     *
     * @return value of autoShow
     */
    public boolean getAutoShow() {
        return getBooleanValue("autoShow");
    }

    /**
     *
     * The State property represents a string that sets the
     * window to be "minimized", "maximized", or "normal" on creation.
     * Default: "normal"
     *
     * @param state value of state
     */
    public void setState(String state) {
        try {
            json.put("state", state);
        } catch (JSONException e) {
            logger.error("Error setting state", e);
        }
    }

    /**
     *
     * The State property represents a string that sets the
     * window to be "minimized", "maximized", or "normal" on creation.
     * Default: "normal"
     *
     * @return value of state
     */
    public String getState() {
        return getString("state");
    }

    /**
     *
     * The Opacity property represents a flag that specifies
     * how transparent the window will be.
     * Default: 1.0
     *
     * @param opacity value of opacity
     */
    public void setOpacity(double opacity) {
        try {
            json.put("opacity", opacity);
        } catch (JSONException e) {
            logger.error("Error setting opacity", e);
        }
    }

    /**
     *
     * The Opacity property represents a flag that specifies
     * how transparent the window will be.
     * Default: 1.0
     *
     * @return value of opacity
     */
    public double getOpacity() {
        return getDoubleValue("opacity");
    }

    /**
     *
     * The CornerRoundingHeight property represents the rounded
     * corners to apply.
     *
     * @param width width of rounded corners
     * @param height height of rounded corners
     */
    public void setCornerRounding(int width, int height) {
        try {
            json.put("cornerRounding", new JSONObject().put("width", width).put("height", height));
        } catch (JSONException e) {
            logger.error("Error setting cornerRounding", e);
        }
    }

    /**
     *
     * The CornerRoundingHeight property represents the rounded
     * corners to apply.
     *
     * @return height of rounded corners
     */
    public int getCornerRoundingHeight() {
        int height = 0;
        JSONObject jsonObject = getJsonValue("cornerRounding");
        if (jsonObject != null) {
            if (jsonObject.has("height")) {
                height = jsonObject.getInt("height");
            }
        }
        return height;
    }

    /**
     *
     * The CornerRoundingHeight property represents the rounded
     * corners to apply.
     *
     * @return width of of rounded corners
     */
    public int getCornerRoundingWidth() {
        int width = 0;
        JSONObject jsonObject = getJsonValue("cornerRounding");
        if (jsonObject != null) {
            if (jsonObject.has("width")) {
                width = jsonObject.getInt("width");
            }
        }
        return width;
    }

    /**
     *
     * The MinWidth property represents the minimum width of a window.
     * Default: 0
     *
     * @param minWidth value of minWidth
     */
    public void setMinWidth(int minWidth) {
        try {
            json.put("minWidth", minWidth);
        } catch (JSONException e) {
            logger.error("Error setting minWidth", e);
        }
    }

    /**
     *
     * The MinWidth property represents the minimum width of a window.
     * Default: 0
     *
     * @return value of minWidth
     */
    public int getMinWidth() {
        return getIntegerValue("minWidth");
    }

    /**
     *
     * The MaxWidth property represents the maximum width  of a window.
     * The value will default to the OS defined value if set to -1.
     *
     * Default: -1
     *
     * @param maxWidth value of maxWidth
     */
    public void setMaxWidth(int maxWidth) {
        try {
            json.put("maxWidth", maxWidth);
        } catch (JSONException e) {
            logger.error("Error setting maxWidth", e);
        }
    }

    /**
     *
     * The MaxWidth property represents the maximum width  of a window.
     * The value will default to the OS defined value if set to -1.
     * Default: -1
     *
     * @return value of maxWidth
     */
    public int getMaxWidth() {
        return getIntegerValue("maxWidth");
    }

    /**
     *
     * The MinHeight property represents the minimum height of a window.
     * Default: 0
     *
     * @param minHeight value of minHeight
     */
    public void setMinHeight(int minHeight) {
        try {
            json.put("minHeight", minHeight);
        } catch (JSONException e) {
            logger.error("Error setting minHeight", e);
        }
    }

    /**
     *
     * The MinHeight property represents the minimum height of a window.
     * Default: 0
     *
     * @return  value of minHeight
     */
    public int getMinHeight() {
        return getIntegerValue("minHeight");
    }

    /**
     *
     * The MaxHeight property represents the maximum height of a window.
     * Will default to the OS defined value if set to -1.
     * Default: -1
     *
     * @param maxHeight value of maxHeight
     */
    public void setMaxHeight(int maxHeight) {
        try {
            json.put("maxHeight", maxHeight);
        } catch (JSONException e) {
            logger.error("Error setting maxHeight", e);
        }
    }

    /**
     *
     * The MaxHeight property represents the maximum height of a window.
     * Will default to the OS defined value if set to -1.
     * Default: -1
     *
     * @return value of maxHeight
     */
    public int getMaxHeight() {
        return getIntegerValue("maxHeight");
    }

    /**
     *
     * The SaveWindowState property defines whether or not a window
     * should maintain a persistent cache of its position/size.
     * Default: true
     *
     * @param saveWindowState value of saveWindowState
     */
    public void setSaveWindowState (boolean saveWindowState) {
        try {
            json.put("saveWindowState", saveWindowState);
        } catch (JSONException e) {
            logger.error("Error setting sateWindowState", e);
        }
    }

    /**
     *
     * The SaveWindowState property defines whether or not a window
     * should maintain a persistent cache of its position/size.
     * Default: true
     *
     * @return value of saveWindowState
     */
    public boolean getSaveWindowState() {
        return getBooleanValue("saveWindowState", true);
    }

    /**
     *
     * The Minimizable property represents a flag that lets the window be minimized.
     * Default: true
     *
     * @param maximizable value of maximizable
     */
    public void setMinimizable(boolean maximizable) {
        try {
            json.put("minimizable", maximizable);
        } catch (JSONException e) {
            logger.error("Error setting minimizable", e);
        }
    }
    public boolean getMinimizable() {
        return getBooleanValue("minimizable");
    }

    /**
     *
     * The Maximizable property represents a flag that lets the window be maximized.
     * Default: true
     *
     * @param maximizable value of maximizable
     */
    public void setMaximizable(boolean maximizable) {
        try {
            json.put("maximizable", maximizable);
        } catch (JSONException e) {
            logger.error("Error setting maximizable", e);
        }
    }

    /**
     *
     * The Maximizable property represents a flag that lets the window be maximized.
     * Default: true
     *
     * @return value of maximizable
     */
    public boolean getMaximizable() {
        return getBooleanValue("maximizable");
    }

    /**
     *
     * The HideOnClose property represents a flag to allow a window
     * to be hidden when the close button is clicked.
     *
     * @param hideOnClose value of hideOnClose
     */
    public void setHideOnClose(boolean hideOnClose) {
        try {
            json.put("hideOnClose", hideOnClose);
        } catch (JSONException e) {
            logger.error("Error setting hideOnClose", e);
        }
    }

    /**
     *
     * The HideOnClose property represents a flag to allow a window
     * to be hidden when the close button is clicked.
     *
     * @return value of hideOnClose
     */
    public boolean getHideOnClose() {
        return getBooleanValue("hideOnClose");
    }

    /**
     *
     * The ContextMenu property represents a flag to show the
     * context menu when right-clicking on a window.
     * Default: false
     *
     * @param contextMenu value of contextMenu
     */
    public void setContextMenu(boolean contextMenu) {
        try {
            json.put("contextMenu", contextMenu);
        } catch (JSONException e) {
            logger.error("Error setting contextMenu", e);
        }
    }

    /**
     *
     * The ContextMenu property represents a flag to show the
     * context menu when right-clicking on a window.
     * Default: false
     *
     * @return  value of contextMenu
     */
    public boolean getContextMenu() {
        return getBooleanValue("contextMenu");
    }

    /**
     *
     * The TaskbarIcon property represents the URL of an icon to be shown on the desktop for
     * a child window.  To set URL of taskbar icon for main window of an app,
     * ApplicationOptions.setApplicationIcon should be called.
     *
     * @param taskbarIcon  value of taskbarIcon
     */
    public void setTaskbarIcon(String taskbarIcon) {
        try {
            json.put("taskbarIcon", taskbarIcon);
        } catch (JSONException e) {
            logger.error("Error setting taskbarIcon", e);
        }
    }

    /**
     *
     * The TaskbarIcon property represents the URL of
     * an icon to be shown on the desktop.
     *
     * @return   value of taskbarIcon
     */
    public String getTaskbarIcon() {
        return getString("taskbarIcon");
    }


    /**
     *
     * The ResizeRegionBottomRightCorner property defines an additional square region located at the bottom
     * right corner of a frameless window.
     *
     * Default value: 4
     *
     * @param resizeRegionBottomRightCorner an additional square region
     */
    public void setResizeRegionBottomRightCorner(int resizeRegionBottomRightCorner) {
        try {
            this.resizeRegionBottomRightCorner = resizeRegionBottomRightCorner;
            json.put("resizeRegion", new JSONObject().put("bottomRightCorner", this.resizeRegionBottomRightCorner)
                    .put("size", this.resizeRegionSize));
        } catch (JSONException e) {
            logger.error("Error setting resizeRegionBottomRightCorner", e);
        }
    }

    /**
     *
     * The ResizeRegionBottomRightCorner property defines an additional square region located at the bottom
     * right corner of a frameless window.
     *
     * @return value of resizeRegionBottomRightCorner, or -1 if it is not set
     */
    public int getResizeRegionBottomRightCorner() {
        int value = -1;
        JSONObject jsonObject = getJsonValue("resizeRegion");
        if (jsonObject != null) {
            if (jsonObject.has("bottomRightCorner")) {
                value = this.getIntegerValue("bottomRightCorner");
            }
        }
        return value;
    }

    /**
     *
     * The ResizeRegionSize property defines a region in pixels that will respond to user mouse interaction for
     * resizing a frameless window
     *
     * Default value: 2
     *
     * @param resizeRegionSize new value of resizeRegionSize
     */
    public void setResizeRegionSize(int resizeRegionSize) {
        try {
            this.resizeRegionSize = resizeRegionSize;
            json.put("resizeRegion", new JSONObject().put("bottomRightCorner", this.resizeRegionBottomRightCorner)
                    .put("size", this.resizeRegionSize));
        } catch (JSONException e) {
            logger.error("Error setting resizeRegionSize", e);
        }
    }

    /**
     *
     * The ResizeRegionSize property defines a region in pixels that will respond to user mouse interaction for
     * resizing a frameless window
     *
     * @return value of resizeRegionSize, or -1 if it is not set
     */
    public int getResizeRegionSize() {
        int value = -1;
        JSONObject jsonObject = getJsonValue("resizeRegion");
        if (jsonObject != null) {
            if (jsonObject.has("size")) {
                value = this.getIntegerValue("size");
            }
        }
        return value;
    }

    /**
     *
     * The DefaultCentered property specifies that the window will be positioned in the
     * center of the primary monitor when loaded for the first time on a machine.
     * Default: false
     *
     * @param defaultCentered new value of defaultCentered
     */
    public void setDefaultCentered(boolean defaultCentered) {
        try {
            json.put("defaultCentered", defaultCentered);
        } catch (JSONException e) {
            logger.error("Error setting defaultCentered", e);
        }
    }

    /**
     *
     * The DefaultCentered property specifies that the window will be positioned in the
     * center of the primary monitor when loaded for the first time on a machine.
     * Default: false
     *
     * @return value of defaultCentered
     */
    public boolean getDefaultCentered() {
        return getBooleanValue("defaultCentered");
    }
    
    /**
     * Get the URL of the preload script that will be retrieved (and cached, if needed) and eval'ed before any other script. Default: inherited from the parent application.
     *
     * @return url of the preload script
     */
    public String getPreload() {
        return getString("preload");
    }

    /**
     * Set A URL of the preload script that will be retrieved (and cached, if needed) and eval'ed before any other script. Default: inherited from the parent application.
     *
     * @param url of the preload script
     */
    public void setPreload(String url) {
        try {
            json.put("preload", url);
        } catch (JSONException e) {
            logger.error("Error setting preload URL", e);
        }
    }
    
    /**
    *
    * The disableIabSecureLogging property represents a flag to disable secure logging in IAB
    * Default: false
    *
    * @return value of disableIabSecureLogging
    */
    public boolean isDisableIabSecureLogging() {
    	return getBooleanValue("disableIabSecureLogging");
    }
    
    /**
    *
    * The disableIabSecureLogging property represents a flag to disable secure logging in IAB
    * Default: false
    *
    * @param disable new value of disableIabSecureLogging
    */
    public void setDisableIabSecureLogging(boolean disable) {
        try {
            json.put("disableIabSecureLogging", disable);
        } catch (JSONException e) {
            logger.error("Error setting disableIabSecureLogging", e);
        }
    }

    /**
     * Set keyboard shortcut options.
     * @param acceleratorOptions Keyboard shortcuts options for devtools, zoom, reload, and reload ignoring cache.
     */
	public void setAccelerator(AcceleratorOptions acceleratorOptions) {
		this.acceleratorOptions = acceleratorOptions;
	}
	
	/**
	 * Get keyboard shortcut options.
	 * @return Keyboard shortcuts options
	 */
	public AcceleratorOptions getAccelerator() {
		if (this.acceleratorOptions == null && this.json.has("accelerator")) {
			this.acceleratorOptions = new AcceleratorOptions(this.json.getJSONObject("accelerator"));
		}
		return this.acceleratorOptions;
	}

	
	public LayoutOptions getLayoutOptions() {
		if (this.layoutOptions == null && this.json.has("layout")) {
			this.layoutOptions = new LayoutOptions(this.json.getJSONObject("layout"));
		}
		return this.layoutOptions;
	}
	
	public void setLayoutOptions(LayoutOptions layoutOptions) {
		this.layoutOptions = layoutOptions;
	}

	@Override
	public JSONObject getJson() {
		this.setJsonBean("accelerator", this.acceleratorOptions);
		this.setJsonBean("layout", this.layoutOptions);
		return super.getJson();
	}

}
