package com.openfin.desktop;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletionStage;

import org.json.JSONArray;
import org.json.JSONObject;

public abstract class WebContent {

	protected DesktopConnection connection;
	protected Identity identity;

	protected WebContent(Identity identity, DesktopConnection connection) {
		this.identity = identity;
		this.connection = connection;
	}

	public Identity getIdentity() {
		return this.identity;
	}

	/**
	 * Gets UUID
	 * 
	 * @return UUID
	 */
	public String getUuid() {
		return this.identity.getUuid();
	}

	/**
	 * Gets name
	 * 
	 * @return name
	 */
	public String getName() {
		return this.identity.getName();
	}

	protected abstract String getEventTopicName();

	public CompletionStage<Ack> addEventListener(String type, EventListener listener) {
		JSONObject eventListenerPayload = this.identity.getJsonCopy();
		eventListenerPayload.put("topic", getEventTopicName());
		eventListenerPayload.put("type", type);
		return this.connection.addEventCallbackAsync(eventListenerPayload, listener, this);
	}

	public CompletionStage<Ack> removeEventListener(String type, EventListener listener) {
		JSONObject eventListenerPayload = this.identity.getJsonCopy();
		eventListenerPayload.put("topic", getEventTopicName());
		eventListenerPayload.put("type", type);
		return this.connection.addEventCallbackAsync(eventListenerPayload, listener, this);
	}

	/**
	 * Get DesktopConnection of this window
	 * 
	 * @return DesktopConnection
	 */
	public DesktopConnection getConnection() {
		return this.connection;
	}

	public CompletionStage<JSONObject> executeJavaScript(String code) {
		JSONObject payload = this.identity.getJsonCopy();
		payload.put("code", code);
		return this.connection.sendActionAsync("execute-javascript-in-window", payload, this).thenApplyAsync(ack -> {
			if (ack.isSuccessful()) {
				return ack.getJsonObject().optJSONObject("data");
			}
			else {
				throw new RuntimeException("error executeJavaScript, reason: " + ack.getReason());
			}
		});
	}

	public CompletionStage<Double> getZoomLevel() {
		return this.connection.sendActionAsync("get-zoom-level", this.identity.getJson(), this).thenApplyAsync(ack -> {
			if (ack.isSuccessful()) {
				return ack.getJsonObject().optDouble("data", 0);
			}
			else {
				throw new RuntimeException("error getZoomLevel, reason: " + ack.getReason());
			}
		});
	}

	public CompletionStage<Void> setZoomLevel(double zoom) {
		JSONObject payload = this.identity.getJsonCopy();
		payload.put("level", zoom);
		return this.connection.sendActionAsync("set-zoom-level", payload, this).thenAcceptAsync(ack -> {
			if (!ack.isSuccessful()) {
				throw new RuntimeException("error setZoomLevel, reason: " + ack.getReason());
			}
		});
	}

	public CompletionStage<Void> navigate(String url) {
		JSONObject payload = this.identity.getJsonCopy();
		payload.put("url", url);
		return this.connection.sendActionAsync("navigate-window", payload, this).thenAcceptAsync(ack -> {
			if (!ack.isSuccessful()) {
				throw new RuntimeException("error navigate, reason: " + ack.getReason());
			}
		});
	}

	public CompletionStage<Void> navigateBack() {
		return this.connection.sendActionAsync("navigate-window-back", this.identity.getJson(), this)
				.thenAcceptAsync(ack -> {
					if (!ack.isSuccessful()) {
						throw new RuntimeException("error navigateBack, reason: " + ack.getReason());
					}
				});
	}

	public CompletionStage<Void> navigateForward() {
		return this.connection.sendActionAsync("navigate-window-forward", this.identity.getJson(), this)
				.thenAcceptAsync(ack -> {
					if (!ack.isSuccessful()) {
						throw new RuntimeException("error navigateForward, reason: " + ack.getReason());
					}
				});
	}

	public CompletionStage<Void> stopNavigation() {
		return this.connection.sendActionAsync("stop-window-navigation", this.identity.getJson(), this)
				.thenAcceptAsync(ack -> {
					if (!ack.isSuccessful()) {
						throw new RuntimeException("error stopNavigation, reason: " + ack.getReason());
					}
				});
	}

	public CompletionStage<Void> reloadAsync(boolean ignoreCache) {
		JSONObject payload = this.identity.getJsonCopy();
		payload.put("ignoreCache", ignoreCache);
		return this.connection.sendActionAsync("reload-window", payload, this).thenAcceptAsync(ack -> {
			if (!ack.isSuccessful()) {
				throw new RuntimeException("error reloadAsync, reason: " + ack.getReason());
			}
		});
	}

	public CompletionStage<Void> printAsync(PrintOptions opts) {
		JSONObject payload = this.identity.getJsonCopy();
		if (opts != null) {
			payload.put("options", opts.getJson());
		}
		return this.connection.sendActionAsync("print", payload, this).thenAcceptAsync(ack -> {
			if (!ack.isSuccessful()) {
				throw new RuntimeException("error printAsync, reason: " + ack.getReason());
			}
		});
	}

	public CompletionStage<Void> findInPage(String searchTerm, FindInPageOptions opts) {
		JSONObject payload = this.identity.getJsonCopy();
		payload.put("searchTerm", searchTerm);
		if (opts != null) {
			payload.put("options", opts.getJson());
		}
		return this.connection.sendActionAsync("find-in-page", payload, this).thenAcceptAsync(ack -> {
			if (!ack.isSuccessful()) {
				throw new RuntimeException("error findInPage, reason: " + ack.getReason());
			}
		});
	}

	public CompletionStage<Void> stopFindInPage(String action) {
		JSONObject payload = this.identity.getJsonCopy();
		payload.put("action", action);
		return this.connection.sendActionAsync("stop-find-in-page", payload, this).thenAcceptAsync(ack -> {
			if (!ack.isSuccessful()) {
				throw new RuntimeException("error stopFindInPage, reason: " + ack.getReason());
			}
		});
	}

	public CompletionStage<List<PrinterInfo>> getPrinters() {
		return this.connection.sendActionAsync("get-printers", this.identity.getJson(), this).thenApplyAsync(ack -> {
			if (ack.isSuccessful()) {
				JSONArray printersJson = ack.getJsonObject().getJSONArray("data");
				int printerCnt = printersJson.length();
				ArrayList<PrinterInfo> printers = new ArrayList<>(printerCnt);
				for (int i = 0; i < printerCnt; i++) {
					printers.add(new PrinterInfo(printersJson.getJSONObject(i)));
				}
				return printers;
			}
			else {
				throw new RuntimeException("error getPrinters, reason: " + ack.getReason());
			}
		});
	}

	public CompletionStage<Void> focusAsync() {
		JSONObject payload = this.identity.getJsonCopy();
		payload.put("emitSynthFocused", true);
		return this.connection.sendActionAsync("focus-window", payload, this).thenAcceptAsync(ack -> {
			if (!ack.isSuccessful()) {
				throw new RuntimeException("error focusAsync, reason: " + ack.getReason());
			}
		});
	}

	public CompletionStage<Void> showDeveloperTools() {
		return this.connection.sendActionAsync("show-developer-tools", this.identity.getJson(), this)
				.thenAcceptAsync(ack -> {
					if (!ack.isSuccessful()) {
						throw new RuntimeException("error showDeveloperTools, reason: " + ack.getReason());
					}
				});
	}
}
