package com.openfin.desktop;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Configuration for launching OpenFin Runtime
 *
 * Created by wche on 3/26/2016.
 */
public class RuntimeConfiguration extends JSONObject {

    private int devToolsPort;
    private String runtimeVersion;
    private String runtimeFallbackVersion;
    private String securityRealm;
    private String additionalRuntimeArguments;
    private JSONObject startupApp;
    private List<JSONObject> appAssets;
    private String rdmURL;
    private String runtimeAssetURL;
    private String additionalRvmArguments;
    private String licenseKey;
    private Boolean nonPersistent;
    private Map<String, Object> configMap = new HashMap<String, Object>();  // used to add any random config setting to app json file
    private String launchRVMPath;  // not null -> launch RVM instead of installer
    private String manifestLocation; // URL of app manifest
    private String localManifestFileName;  // name of automatically generated local manifest file
    private String generatedManifestLocation;  // if manifestLocation not set, Java adapter generates a manifest to launch Runtime
    private boolean showInstallerUI=false;  // default to silent mode
    private int maxMessageSize = 0; //use jetty default value (64K)
    private boolean useNamedPipePortDiscovery = false;
    private int runtimePort = -1;
    private List<JSONObject> services;
    private boolean doNotLaunch = false;
    private static final String RVM_OPT_DO_NOT_LAUNCH = "--do-not-launch";    
    /**
     * Constructor
     */
    public RuntimeConfiguration() {
        this.appAssets = new ArrayList<JSONObject>();
    }

    /**
     * Set port for accessing devtools on localhost
     * @param port port number
     */
    public void setDevToolsPort(int port) {
        this.devToolsPort = port;
    }

    /**
     * Set version number of Runtime to launch
     * @param version version number
     */
    public void setRuntimeVersion(String version) {
        this.runtimeVersion = version;
    }

    /**
     * Get version number of Runtime
     * @return version of Runtime
     */
    public String getRuntimeVersion() {
        return this.runtimeVersion;
    }

    /**
     * Get fallback version number of Runtime to launch
     *
     * @return fallback version
     */
    public String getRuntimeFallbackVersion() {
        return runtimeFallbackVersion;
    }

    /**
     * Set fallback version number of Runtime to launch
     *
     * @param runtimeFallbackVersion fallback version of Runtime
     */
    public void setRuntimeFallbackVersion(String runtimeFallbackVersion) {
        this.runtimeFallbackVersion = runtimeFallbackVersion;
    }

    /**
     * Set security realm of Runtime
     *
     * @param securityRealm name of security realm
     */
    public void setSecurityRealm(String securityRealm) {
        this.securityRealm = securityRealm;
    }

    /**
     * Get security realm
     * @return security realm
     */
    public String getSecurityRealm() {
        return this.securityRealm;
    }

    /**
     * Set additional arguments for Runtime
     *
     * @param additionalRuntimeArguments additional arguments
     */
    public void setAdditionalRuntimeArguments(String additionalRuntimeArguments) {
        this.additionalRuntimeArguments = additionalRuntimeArguments;
    }

    /**
     * Add configuration of an app asset
     *
     * @param assetConfig configuration in JSON format
     */
    public void addAppAsset(JSONObject assetConfig) {
        this.appAssets.add(assetConfig);
    }

    /**
     * Set URL of RDM service
     *
     * @param rdmURL RDM URL
     * @deprecated more info at https://openfin.co/documentation/desktop-owner-settings/
     */
    public void setRdmURL(String rdmURL) {
        this.rdmURL = rdmURL;
    }

    /**
     * Get URL of RDM service
     *
     * @return URL of RDM service
     * @deprecated more info at https://openfin.co/documentation/desktop-owner-settings/
     */
    public String getRdmURL() {
        return this.rdmURL;
    }

    /**
     * Set URL of Runtime assets, including RVM and Runtime
     *
     * @param runtimeAssetURL URL of Runtime assets
     * @deprecated more info at https://openfin.co/documentation/desktop-owner-settings/
     */
    public void setRuntimeAssetURL(String runtimeAssetURL) {
        this.runtimeAssetURL = runtimeAssetURL;
    }

    /**
     * Get URL of Runtime assets
     *
     * @return URL of Runtime assets
     * @deprecated more info at https://openfin.co/documentation/desktop-owner-settings/
     */
    public String getRuntimeAssetURL() {
        return this.runtimeAssetURL;
    }

    /**
     * Get additional arguments for RVM and Installer
     *
     * @return additional arguments
     */
    public String getAdditionalRvmArguments() {
    	if (this.doNotLaunch) {
    		if (this.additionalRvmArguments == null) {
    			this.additionalRvmArguments = RVM_OPT_DO_NOT_LAUNCH;
    		}
    		else if (!this.additionalRvmArguments.contains(RVM_OPT_DO_NOT_LAUNCH)) {
    			this.additionalRvmArguments = this.additionalRvmArguments + " " + RVM_OPT_DO_NOT_LAUNCH;
    		}
    	}
    	return additionalRvmArguments;
    }

    /**
     * set Defines the minimum RVM version the app requires. This option is available in RVM 2.3 or greater
     *
     * @param rvmVersion version of RVM required
     */
    public void setRvmVersion(String rvmVersion) {
        this.addConfigurationItem("rvmVersion", rvmVersion);
    }

    /**
     * set configuration for shortcut
      * @param shortCut config for shortcut
     */
    public void setShortCut(JSONObject shortCut) {
        this.addConfigurationItem("shortcut", shortCut);
    }

    /**
     * set configuration to customize the appearance of the RVM progress dialog
     *
     * @param dialogSettings config for dialog
     */
    public void setDialogSettings(JSONObject dialogSettings) {
        this.addConfigurationItem("dialogSettings", dialogSettings);
    }

    /**
     * Specify an image to display while the runtime is loading. It takes any image file (including semi-transparent PNGs)
     *
     * @param splashScreenImage URL of the image
     */
    public void setSplashScreenImage(String splashScreenImage) {
        this.addConfigurationItem("splashScreenImage", splashScreenImage);
    }

    /**
     * Specify customized error messages during launch
     *
     * @param supportInformation config for support info
     */
    public void setSupportInformation(String supportInformation) {
        this.addConfigurationItem("supportInformation", supportInformation);
    }

    /**
     * Set additional arguments for RVM and Installer
     *
     * @param additionalRvmArguments set additional arguments
     */
    public void setAdditionalRvmArguments(String additionalRvmArguments) {
        this.additionalRvmArguments = additionalRvmArguments;
    }

    /**
     * Set configuration of startup application.
     *
     * @param startupApp configuration in JSON format
     */
    public void setStartupApp(JSONObject startupApp) {
        this.startupApp = startupApp;
    }

    /**
     * Get license key for Runtime
     *
     * @return license key
     */
    public String getLicenseKey() {
        return licenseKey;
    }

    /**
     * Set license key for Runtime
     *
     * @param licenseKey license key
     */
    public void setLicenseKey(String licenseKey) {
        this.licenseKey = licenseKey;
    }

    /**
     * Get value of nonPersistent
     *
     * @return value of nonPersistent
     */
    public boolean isNonPersistent() {
        return nonPersistent != null && nonPersistent;
    }

    /**
     * Automatically disconnect the adapter from the runtime when the last application or persistent connection closes
     * Defaults to false
     *
     * @param nonPersistent true for non persistent connection
     */
    public void setNonPersistent(Boolean nonPersistent) {
        this.nonPersistent = nonPersistent;
    }

    /**
     * RVM option, to perform all of the startup sequence steps but do not launch the app
     * @param doNotLaunch true to add --do-not-launch options for RVM
     */
    public void setDoNotLaunch(boolean doNotLaunch) {
    	this.doNotLaunch = doNotLaunch;
    }

    /**
     * Get value of RVM option doNotLaunch setting
     * @return value of doNotLaunch
     */
    public boolean isDoNotLaunch() {
    	if (this.additionalRvmArguments != null && this.additionalRvmArguments.contains(RVM_OPT_DO_NOT_LAUNCH)) {
    		return true;
    	}
    	else {
        	return this.doNotLaunch;
    	}
    }

    public boolean isUseNamedPipePortDiscovery() {
        return useNamedPipePortDiscovery;
    }

    public void setUseNamedPipePortDiscovery(boolean useNamedPipePortDiscovery) {
        this.useNamedPipePortDiscovery = useNamedPipePortDiscovery;
    }

    /**
     * Add a configuration setting for launching Runtime.  value can be type of Integer, Long, Double, String or JSONObject
     *
     * @param key key value of the setting
     * @param value value of the setting
     */
    public void addConfigurationItem(String key, Object value) {
        this.configMap.put(key, value);
    }

    /**
     * By default, Java adapter launches Runtime with OpenFin installer.  If RVM path is set, installer is skipped and RVM is started instead.
     *
     * @param path path for RVM
     */
    public void setLaunchRVMPath(String path) {
        this.launchRVMPath = path;
    }

    /**
     * return value of launchRVMPath
     *
     * @return path
     */
    public String getLaunchRVMPath() {
        return this.launchRVMPath;
    }

    /**
     * Get location of app manifest.  If manifest is set, some fields, such as runtimeVersion, will be copied from the manifest
     *
     * @return location of app manifest
     */
    public String getManifestLocation() {
        return manifestLocation;
    }

    String getGeneratedManifestLocation() {
        return this.generatedManifestLocation;
    }

    void setGeneratedManifestLocation(String manifestLocation) {
        this.generatedManifestLocation = manifestLocation;
    }

    /**
     * Get name of local manifest file name
     *
     * @return file name
     */
    public String getLocalManifestFileName() {
        return localManifestFileName;
    }

    /**
     * Set name of automatically generated local manifest file.  Only name of the file, not path, should be specified.
     * An extension of ".json" is appended.
     *
     * @param localManifestFileName file name
     */
    public void setLocalManifestFileName(String localManifestFileName) {
        this.localManifestFileName = localManifestFileName;
    }

    /**
     * Get if OpenFinInstaller should be invoked with UI
     * @return boolean value of showInstallUI
     * @deprecated
     */
    public boolean isShowInstallerUI() {
        return this.showInstallerUI;
    }

    /**
     * Set if OpenFinInstaller should be invoked with UI.  Defaults to false
     *
     * @param showInstallerUI value of showInstallerUI
     *
     */
    public void setShowInstallerUI(boolean showInstallerUI) {
        this.showInstallerUI = showInstallerUI;
    }

    /**
     * Set location of app manifest
     *
     * @param manifestLocation location of app manifest
     */
    public void setManifestLocation(String manifestLocation) {
        this.manifestLocation = manifestLocation;
    }

    /**
     * Generates config for "runtime" section
     */
    private void appendRuntimeOptions(JSONObject rootConfig) {
        JSONObject runtimeConfig = new JSONObject();
        StringBuffer arguments = new StringBuffer();
        if (this.securityRealm != null) {
            arguments.append(String.format(" --security-realm=%s ", this.securityRealm));
        }
        if (this.additionalRuntimeArguments != null) {
            arguments.append(String.format(" %s ", this.additionalRuntimeArguments));
        }
        runtimeConfig.put("arguments", arguments);
        runtimeConfig.put("version", this.runtimeVersion);
        if (this.runtimeFallbackVersion != null) {
            runtimeConfig.put("fallbackVersion", this.runtimeFallbackVersion);
        }
        rootConfig.put("runtime", runtimeConfig);
    }

    /**
     * Append app assets config to a root configuration
     *
     * @param rootConfig root level configuration
     */
    private void appendAppAssets(JSONObject rootConfig) {
        if (this.appAssets.size() > 0) {
            rootConfig.put("appAssets", this.appAssets);
        }
    }

    /**
     * Append startup_app config to a root configuration
     *
     * @param rootConfig root level configuration
     */
    private void appendStartupConfig(JSONObject rootConfig) {
        if (this.startupApp != null) {
            rootConfig.put("startup_app", this.startupApp);
        }
    }

    private void appendAdditionalConfigItems(JSONObject rootConfig) {
        if (this.configMap.size() > 0) {
            for (String key : this.configMap.keySet()) {
                Object value = this.configMap.get(key);
                if (value instanceof Double) {
                    rootConfig.put(key, ((Double) value).doubleValue());
                }
                else if (value instanceof Integer) {
                    rootConfig.put(key, ((Integer) value).intValue());
                }
                else if (value instanceof Long) {
                    rootConfig.put(key, ((Long) value).longValue());
                } else {
                    rootConfig.put(key, value);
                }
            }
        }
    }

    /**
     * Append rdmUrl config to a root configuration
     *
     * @param rootConfig root level configuration
     */
    private void appendRDMURL(JSONObject rootConfig) {
        if (this.rdmURL != null) {
            rootConfig.put("rdmUrl", this.rdmURL);
        }
    }

    /**
     * Append assetsUrl config to a root configuration
     *
     * @param rootConfig root level configuration
     */
    private void appendRuntimeAssetsURL(JSONObject rootConfig) {
        if (this.runtimeAssetURL != null) {
            rootConfig.put("assetsUrl", this.runtimeAssetURL);
        }
    }

    private void appendLicensekey(JSONObject rootConfig) {
        if (this.licenseKey != null) {
            rootConfig.put("licenseKey", this.licenseKey);
        }
    }

    /**
     * Get max size of each message between Java adapter and Runtime
     *
     * @return max size
     */
    public int getMaxMessageSize() {
		return maxMessageSize;
	}

    /**
     * Set max size of each message between Java adapter and Runtime
     *
     * @param maxMessageSize max size
     */
	public void setMaxMessageSize(int maxMessageSize) {
		this.maxMessageSize = maxMessageSize;
	}

	private void appendMaxMessageSize(JSONObject rootConfig) {
        if (this.maxMessageSize > 0) {
            // @todo need to coordicate with Runtime for setting max size
            //rootConfig.put("maxMessageSize", this.maxMessageSize);
        }
    }

    /**
     * Get port number of Runtime websocket server
     *
     * @return port number
     */
    public int getRuntimePort() {
        return runtimePort;
    }

    /**
     * Set port number of Runtime websocket server
     *
     * @param runtimePort port number
     */
    public void setRuntimePort(int runtimePort) {
        this.runtimePort = runtimePort;
    }
    
    public void addService(String name, String manifestUrl) {
    	if (this.services == null) {
            this.services = new ArrayList<JSONObject>();
    	}
    	JSONObject service = new JSONObject();
    	service.put("name", name);
    	if (manifestUrl != null) {
    		service.put("manifestUrl", manifestUrl);
    	}
    	this.services.add(service);
    }
    
    private void appendServices(JSONObject rootConfig) {
        if (this.services != null) {
            rootConfig.put("services", this.services);
        }
    }


    /**
     * Generates JSON string that can be passed as --config to RVM and Insaller
     *
     * @return JSON string
     */
    public String generateRuntimeConfig() {
        JSONObject config = new JSONObject();
        if (this.devToolsPort > 0) {
            config.put("devtools_port", this.devToolsPort);
        }
        appendRuntimeOptions(config);
        appendRDMURL(config);
        appendRuntimeAssetsURL(config);
        appendAppAssets(config);
        appendStartupConfig(config);
        appendServices(config);
        appendAdditionalConfigItems(config);
        appendLicensekey(config);
        appendMaxMessageSize(config);
        return config.toString();
    }


}
