package com.openfin.desktop;

import com.openfin.desktop.win32.InstallChecker;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;
import java.util.concurrent.CompletableFuture;

/**
 *     An object representing the core of the OpenFin Runtime. Allows the developer to
 *     perform system-level actions, such as accessing logs, viewing processes, clearing
 *     the cache and exiting the Runtime.
 *
 * Created by wche on 1/26/16.
 *
 */
public class OpenFinRuntime {
    private static Logger logger = LoggerFactory.getLogger(OpenFinRuntime.class.getName());

    protected DesktopConnection connection;
    protected JSONObject eventListenerPayload, configPayload;

    private static String adapterVersion;
    private static long adapterBuildTime;

    static {
        logger = LoggerFactory.getLogger(OpenFinRuntime.class.getName());

        // get adapter version info
        try {
            Package aPackage = OpenFinRuntime.class.getPackage();
            adapterVersion = aPackage.getImplementationVersion();
        } catch (Exception ex) {
            adapterVersion = null;
            logger.error("Error reading version", ex);
        }

        // get adapter build time
        try {
            String rn = OpenFinRuntime.class.getName().replace('.', '/') + ".class";
            URL url = ClassLoader.getSystemResource(rn);
            if (url != null) {
                URLConnection conn = url.openConnection();
                if (conn != null) {
                    if (conn instanceof JarURLConnection) {
                        JarURLConnection j = (JarURLConnection) conn;
                        adapterBuildTime = j.getJarFile().getEntry("META-INF/MANIFEST.MF").getTime();
                    } else {
                        logger.info("OpenFin Java Adapter system resource connection " + conn.getClass().getName());
                    }
                } else {
                    logger.info("OpenFin Java Adapter null system resource connection " + rn);
                }
            } else {
                logger.info("OpenFin Java Adapter null system resource " + rn);
            }
        } catch (Exception ex) {
            adapterBuildTime = 0;
            logger.error("Error reading buildTime");
        }

        logger.info("OpenFin Java Adapter version " + adapterVersion);
        logger.info("OpenFin Java Adapter built time " + adapterBuildTime);

    }

    /**
     * Constructor
     * @param connection Connection object to the AppDesktop
     * @see DesktopConnection
     */
    public OpenFinRuntime(DesktopConnection connection) {
        this.connection = connection;
    }

    /**
     * Gets Device ID
     * @param listener AckListener for device ID
     * @see AckListener
     */
    public void getDeviceId(AckListener listener) {
        connection.sendAction("get-device-id", null, listener, this);
    }
    
    /**
     * Returns a frame info object relating to the entity specified by the uuid and name passed in. 
     * The possible types are 'window', 'iframe', 'external connection' or 'unknown'.
     * @param uuid entity uuid
     * @param name entity name
     * @param listener AckListener for entity info
     * @see AckListener
     */
    public void getEntityInfo(String uuid, String name, AckListener listener) {
    	JSONObject payload = new JSONObject();
    	payload.put("uuid", uuid);
    	payload.put("name", name);
    	connection.sendAction("get-entity-info", payload, listener, this);
    }

    /**
     * Gets AppDesktop version number
     * @param listener AckListener for version number
     * @see AckListener
     */
    public void getVersion(AckListener listener) {
        connection.sendAction("get-version", null, listener, this);
    }

    /**
     * Retrieves the command line argument string that started App Desktop
     * @param listener AckListener for command line argument
     * @see AckListener
     */
    public void getCommandLineArguments(AckListener listener) {
        connection.sendAction("get-command-line-arguments", null, listener, this);
    }

    /**
     * Retrieves an array of all App Desktop processes that are currently running
     * Each element in the array is an object containing the uuid
     * and the name of the application to which the process belongs.
     * @param listener AckListener for process list
     * @see AckListener
     */
    public void getProcessList(AckListener listener) {
        connection.sendAction("process-snapshot", null, listener, this);
    }

    /**
     * Retrieves the contents of the log with the specified filename
     * @param logName The filename of the log
     * @param listener AckListener for log contents
     * @see AckListener
     */
    public void getLog(String logName, AckListener listener) {
        try {
            connection.sendAction("view-log", new JSONObject().put("name", logName), listener, this);
        } catch (JSONException e) {
            logger.error("Error getLog", e);
        }
    }

    /**
     * Retrieves an array of data objects for all available logs
     * Each object in the returned array takes the form:
     *   {
     *       name: (string) the filename of the log,
     *       size: (integer) the size of the log in bytes,
     *       date: (integer) the unix time at which the log was created
     *    }
     * @param listener AckListener for log list
     * @see AckListener
     */
    public void getLogList(AckListener listener) {
        connection.sendAction("list-logs", null, listener, this);
    }

    /**
     * Writes a message to the log
     *
     * @param level The log level for the entry. Can be either "info", "warning" or "error"
     * @param message The log message text
     * @throws DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     */
    public void log(String level, String message) throws DesktopException {
        try {
            connection.sendAction("write-to-log", new JSONObject()
                    .put("level", level)
                    .put("message", message));
        } catch (JSONException e) {
            logger.error("Error logging message", e);
            throw new DesktopException(e);
        }
    }

    /**
     * Writes a message to the log
     * @param level The log level for the entry. Can be either "info", "warning" or "error"
     * @param message The log message text
     * @param listener AckListener for the result
     * @throws DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AckListener
     */
    public void log(String level, String message, AckListener listener) throws DesktopException{
        try {
            connection.sendAction("write-to-log", new JSONObject()
                    .put("level", level)
                    .put("message", message), listener, this);
        } catch (JSONException e) {
            logger.error("Error logging message", e);
            throw new DesktopException(e);
        }
    }

    /**
     * Retrieves the proxy settings object
     * <pre>
     * The proxy object the callback receives takes the following form:
     * {
     *     type: (string) "system" or "named",
     *     proxyAddress: (string) the address of the proxy server,
     *     proxyPort: (integer) the port of proxy server
     * }
     * </pre>
     *
     * @param listener AckListener for the request
     * @see AckListener
     */
    public void getProxySettings(AckListener listener) {
        connection.sendAction("get-proxy-settings", null, listener, this);
    }

    /**
     * Updates the proxy settings
     * The passed type can be either "system" or "named".
     * Use "system" to use the default system proxy settings.
     * Otherwise use "named" to specify the address and port
     * of the proxy server.
     *
     * @param type Type of the proxy
     * @param proxyAddress Address of the proxy
     * @param proxyPort Port of the proxy
     * @param listener AckListener for the request
     * @throws DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AckListener
     */
    public void updateProxySettings(String type, String proxyAddress, int proxyPort, AckListener listener) throws DesktopException {
        try {
            connection.sendAction("update-proxy",
                    new JSONObject()
                            .put("type", type)
                            .put("proxyAddress", proxyAddress)
                            .put("proxyPort", proxyPort), listener, this);
        } catch (JSONException e) {
            logger.error("Error updating proxy settings", e);
            throw new DesktopException(e);
        }
    }

    /**
     *  Clears cached data containing window state/positions,
     *  application resource files (images, HTML, JavaScript files)
     *  cookies, and items stored in the Local Storage.
     *
     * @param cache If true, clears chrome caches
     * @param cookies If true, deletes all cookies
     * @param localStorage If true, clear application caches
     * @param appcache If true, clears local storage
     * @param userData If true, clears user data
     * @throws DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     *
     */
    public void clearCache(boolean cache,
                           boolean cookies,
                           boolean localStorage,
                           boolean appcache,
                           boolean userData) throws DesktopException {
        try {
            connection.sendAction("clear-cache",
                    new JSONObject()
                            .put("cache", cache)
                            .put("cookies", cookies)
                            .put("localStorage", localStorage)
                            .put("appcache", appcache)
                            .put("userData", userData));
        } catch (JSONException e) {
            logger.error("Error clearing cache", e);
            throw new DesktopException(e);
        }
    }

    /**
     *  Clears cached data containing window state/positions,
     *  application resource files (images, HTML, JavaScript files)
     *  cookies, and items stored in the Local Storage.
     *
     * @param cache If true, clears chrome caches
     * @param cookies If true, deletes all cookies
     * @param localStorage If true, clear application caches
     * @param appcache If true, clears local storage
     * @param userData If true, clears user data
     * @param listener AckListener for the request
     * @throws DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AckListener
     */
    public void clearCache(boolean cache,
                           boolean cookies,
                           boolean localStorage,
                           boolean appcache,
                           boolean userData, AckListener listener) throws DesktopException {
        try {
            connection.sendAction("clear-cache",
                    new JSONObject()
                            .put("cache", cache)
                            .put("cookies", cookies)
                            .put("localStorage", localStorage)
                            .put("appcache", appcache)
                            .put("userData", userData), listener, this);
        } catch (JSONException e) {
            logger.error("Error clearing cache", e);
            throw new DesktopException(e);
        }
    }

    /**
     * Clears all cached data when Runtime is restarted
     *
     * @throws DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     */
    public void deleteCacheOnRestart() throws DesktopException {
        connection.sendAction("delete-cache-request", null);
    }

    /**
     * Clears all cached data when App Desktop is restarted
     * @param listener AckListener for the request
     * @see AckListener
     */
    public void deleteCacheOnRestart(AckListener listener) {
        connection.sendAction("delete-cache-request", null, listener, this);
    }

    /**
     * Opens the passed URL
     * @param listener AckListener for the request
     * @see AckListener
     */
    public void openUrlWithBrowser(AckListener listener) {
        connection.sendAction("open-url-with-browser", null, listener, this);
    }

    /**
     * Retrieves an object that contains data about the about the
     * <pre>
     * monitor setup of the computer that App Desktop is running on.
     *     The returned object takes the form:
     *     {
     *         nonPrimaryMonitors: [{
     *             availableRect: {
     *                 bottom: (integer) bottom-most available monitor coordinate,
     *                 left: (integer) left-most available monitor coordinate,
     *                 right: (integer) right-most available monitor coordinate,
     *                 top: (integer) top-most available monitor coordinate
     *             },
     *             deviceId: (string) device id of the display,
     *             displayDeviceActive: (boolean) true if the display is active,
     *             monitorRect: {
     *                 bottom: (integer) bottom-most monitor coordinate,
     *                 left: (integer) left-most monitor coordinate,
     *                 right: (integer) right-most monitor coordinate,
     *                 top: (integer) top-most monitor coordinate
     *             },
     *             name: (string) name of the display
     *         },
     *         ...
     *         ],
     *         primaryMonitor: {
     *             availableRect: {
     *                 bottom: (integer) bottom-most available monitor coordinate,
     *                 left: (integer) left-most available monitor coordinate,
     *                 right: (integer) right-most available monitor coordinate,
     *                 top: (integer) top-most available monitor coordinate
     *             },
     *             deviceId: (string) device id of the display,
     *             displayDeviceActive: (boolean) true if the display is active,
     *             monitorRect: {
     *                 bottom: (integer) bottom-most monitor coordinate,
     *                 left: (integer) left-most monitor coordinate,
     *                 right: (integer) right-most monitor coordinate,
     *                 top: (integer) top-most monitor coordinate
     *             },
     *             name: (string) name of the display
     *         },
     *         reason: (string) always "api-query",
     *         taskbar: {
     *             edge: {string} which edge of a monitor the taskbar is on,
     *             rect: {
     *                 bottom: ({integer} bottom-most coordinate of the taskbar),
     *                 left: ({integer} left-most coordinate of the taskbar),
     *                 right: ({integer} right-most coordinate of the taskbar),
     *                 top: ({integer} top-most coordinate of the taskbar)
     *             }
     *         },
     *         virtualScreen: {
     *             bottom: (integer) bottom-most coordinate of the virtual screen,
     *             left: (integer) left-most coordinate of the virtual screen,
     *             right: (integer) right-most coordinate of the virtual screen,
     *             top: (integer) top-most coordinate of the virtual screen
     *         }
     *     }
     * </pre>
     *
     * @param listener AckListener for the request
     * @see AckListener
     */
    public void getMonitorInfo(AckListener listener) {
        connection.sendAction("get-monitor-info", null, listener, this);
    }
    
    public CompletableFuture<MonitorInfo> getMonitorInfoAsync() {
    	return connection.sendActionAsync("get-monitor-info", null, this).thenApplyAsync(ack->{
    		if (ack.isSuccessful()) {
        		return new MonitorInfo(ack.getJsonObject().getJSONObject("data"));
    		}
    		else {
    			throw new RuntimeException("error getting monitor info, reason: " + ack.getReason());
    		}
    	});
    }

    /**
     *     The object passed to callback takes the form:
     *     <pre>
     *     [
     *         {
     *             uuid: (string) uuid of the application,
     *             mainWindow: {
     *                 name: (string) name of the main window,
     *                 top: (integer) top-most coordinate of the main window,
     *                 right: (integer) right-most coordinate of the main window,
     *                 bottom: (integer) bottom-most coordinate of the main window,
     *                 left: (integer) left-most coordinate of the main window
     *             },
     *             childWindows: [{
     *                     name: (string) name of the child window,
     *                     top: (integer) top-most coordinate of the child window,
     *                     right: (integer) right-most coordinate of the child window,
     *                     bottom: (integer) bottom-most coordinate of the child window,
     *                     left: (integer) left-most coordinate of the child window
     *                 },
     *                 ...
     *             ]
     *         },
     *         ...
     *     ]
     * </pre>
     *
     * @param listener AckListener for the request
     * @see AckListener
     */
    public void getAllWindows(AckListener listener) {
        connection.sendAction("get-all-windows", null, listener, this);
    }

    /**
     * Retrieves an array of data (uuid, running/active state) for all application windows
     * <pre>
     *     The object passed to callback takes the form:
     *     [
     *         {
     *             uuid: (string) uuid of the application,
     *             isRunning: (bool) true when the application is running/active
     *         },
     *         ...
     *     ]
     * </pre>
     *
     * @param listener AckListener for the request
     */
    public void getAllApplications(AckListener listener) {
        this.connection.sendAction("get-all-applications", new JSONObject(), listener, this);
    }


    /**
     * Exit Runtime
     *
     * @throws DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     */
    public void exit() throws DesktopException {
        connection.exit();
    }

    /**
     * Exits App Desktop
     * @param listener AckListener for the request
     * @see AckListener
     * @deprecated use exit() instead.
     *
     */
    @Deprecated
    public void exit(AckListener listener) {
        try {
            exit();
        } catch (Exception e) {
        }
    }

    /**
     * Returns the mouse in virtual screen coordinates (left, top)
     * <pre>
     *     The returned object takes the form:
     *     {
     *         top: (integer) the top position of the mouse in virtual screen
     *                        coordinates,
     *         left: (integer) the left position of the mouse in virtual screen
     *                         coordinates
     *     }
     * </pre>
     *
     * @param listener AckListener for the request
     * @see AckListener
     */
    public void getMousePosition(AckListener listener) {
        connection.sendAction("get-mouse-position", null, listener, this);
    }

    /**
     * Opens the passed URL
     *
     * @throws DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     */
    public void openUrlWithBrowser() throws DesktopException {
        connection.sendAction("open-url-with-browser", null);
    }

    /**
     * Retrieves the Runtime's configuration
     *
     * @param section Which section to return from the configuration.  Pass null to get all sections
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to get Runtime configuration
     * @see DesktopException
     * @see AckListener
     */
    public void getConfig(String section, AckListener callback) throws DesktopException{
        try {
            if (configPayload == null) {
                configPayload = new JSONObject();
            }
            configPayload.put("section", section);
            this.connection.sendAction("get-config", configPayload, callback, this);
        } catch (Exception e) {
            logger.error("Error setConfig", e);
            throw new DesktopException(e);
        }
    }

    /**
     * Shows Developer tool
     * @param applicationUUID The application ID
     * @param windowName The name of dev tool window
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to show devtools
     * @see DesktopException
     * @see AckListener
     */
    public void showDeveloperTools(String applicationUUID, String windowName, AckListener callback) throws DesktopException{
        try {
            JSONObject developerToolsPayload = new JSONObject();
            developerToolsPayload.put("uuid", applicationUUID);
            developerToolsPayload.put("name", windowName);
            this.connection.sendAction("show-developer-tools", developerToolsPayload, callback, this);
        } catch (Exception e) {
            logger.error("Error showing developer tool", e);
            throw new DesktopException(e);
        }
    }


    /**
     * Registers an event listener on the specified event
     * <pre>
     *     Supported system event types are:
     *          desktop-icon-clicked
     *          idle-state-changed
     *          monitor-info-changed
     *          session-changed
     * </pre>
     *
     * @param subscriptionObject A JSON object containing subscription information such as the topic and type
     * @param listener EventListener for the event
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to add event listener specified
     * @see EventListener
     * @see AckListener
     */
    protected void addEventListener(JSONObject subscriptionObject,
                                    EventListener listener,
                                    AckListener callback) throws DesktopException {
        this.connection.addEventCallback(subscriptionObject, listener, callback, this);
    }

    /**
     * Registers an event listener on the specified event
     * <pre>
     *     Supported system event types are:
     *          desktop-icon-clicked
     *          idle-state-changed
     *          monitor-info-changed
     *          session-changed
     * </pre>
     *
     * @param type Type of the event
     * @param listener EventListener for the event
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to add event listener specified
     * @see EventListener
     * @see AckListener
     */
    public void addEventListener(String type, EventListener listener, AckListener callback) throws DesktopException {
        try {
            if (eventListenerPayload == null) {
                eventListenerPayload = new JSONObject();
                eventListenerPayload.put("topic", "system");
            }
            eventListenerPayload.put("type", type);
            addEventListener(eventListenerPayload, listener, callback);
        } catch (Exception e) {
            logger.error("Error adding event listener", e);
            throw new DesktopException(e);
        }
    }

    /**
     * Removes a previously registered event listener from the specified event
     *
     * @param type Type of the event
     * @param listener EventListener for the event
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to remove event listener specified
     * @see DesktopException
     * @see EventListener
     * @see AckListener
     */
    public void removeEventListener(String type, EventListener listener, AckListener callback) throws DesktopException{
        try {
            if (eventListenerPayload == null) {
                eventListenerPayload = new JSONObject();
                eventListenerPayload.put("topic", "system");
            }
            eventListenerPayload.put("type", type);
            this.connection.removeEventCallback(eventListenerPayload, listener, callback, this);
        } catch (Exception e) {
            logger.error("Error reoving event listener", e);
            throw new DesktopException(e);
        }
    }

    /**
     *
     * Runs an executable or batch file.
     *
     * @param path The path of the file to launch via the command line
     * @param commandLine The command line arguments to pass
     * @param callback A function that is called if the method succeeds
     * @param listener A function that is called if the method fails
     * @throws  DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AsyncCallback
     * @see LaunchExternalProcessResult
     * @see AckListener
     */
    public void launchExternalProcess(String path, String commandLine, final AsyncCallback<LaunchExternalProcessResult> callback, final AckListener listener) throws DesktopException {
        JSONObject launchExternalProcessPayload = new JSONObject();

        try {
            launchExternalProcessPayload.put("path", path);
            launchExternalProcessPayload.put("arguments", commandLine);
            launchExternalProcessPayload.put("commandLine", commandLine);  // backwards compatible
            this.launchExternalProcess(launchExternalProcessPayload, callback, listener);
        } catch (Exception e) {
            logger.error("Error launching external process", e);
            throw new DesktopException(e);
        }
    }

    /**
     *
     * Runs an executable or batch file.
     *
     * @param launchConfig configuration for launching external process
     * @param callback A function that is called if the method succeeds
     * @param listener A function that is called if the method fails
     * @throws  DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AsyncCallback
     * @see LaunchExternalProcessResult
     * @see AckListener
     */
    public void launchExternalProcess(JSONObject launchConfig, final AsyncCallback<LaunchExternalProcessResult> callback, final AckListener listener) throws DesktopException {
        try {
            AckListener mainCallback = null;
            if (callback != null) {
                mainCallback = new AckListener() {
                    @Override
                    public void onSuccess(Ack ack) {
                        try {
                            JSONObject data = (JSONObject) ack.getData();
                            callback.onSuccess(new LaunchExternalProcessResult(data.getString("uuid")));
                        } catch (Exception e) {
                            logger.error("Error processing result from launching external process", e);
                        }
                    }

                    @Override
                    public void onError(Ack ack) {
                        DesktopUtils.errorAck(listener, ack);
                    }
                };
            }
            this.connection.sendAction("launch-external-process", launchConfig, mainCallback, this);
        } catch (Exception e) {
            logger.error("Error launching external process", e);
            throw new DesktopException(e);
        }
    }

    /**
     *
     * Attempts to cleanly close an external process and terminates it
     * if the close has not occured after the elapsed timeout in milliseconds.
     *
     * @param processUuid The UUID for a process launched by DesktopSystem.launchExternalProcess()
     * @param timeout The time in milliseconds to wait for a close to occur before terminating
     * @param killTree true if child processes are included
     * @param callback A function that is called if the method succeed with result code being passed
     * @param listener A function that is called if the method fails
     * @throws  DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AsyncCallback terminateExternalProcess
     * @see TerminateExternalProcessResult
     * @see AckListener
     */
    public void terminateExternalProcess(final String processUuid, int timeout, boolean killTree, final AsyncCallback<TerminateExternalProcessResult> callback, final AckListener listener) throws DesktopException {
        try {
            JSONObject terminateExternalProcessPayload = new JSONObject();
            terminateExternalProcessPayload.put("uuid", processUuid);
            terminateExternalProcessPayload.put("timeout", timeout);
            terminateExternalProcessPayload.put("child", killTree);

            AckListener mainCallback = null;
            if (callback != null) {
                mainCallback = new AckListener() {
                    @Override
                    public void onSuccess(Ack ack) {
                        try {
                            JSONObject data = (JSONObject) ack.getData();
                            callback.onSuccess(new TerminateExternalProcessResult(processUuid, data.getString("result")));
                        } catch (Exception e) {
                            logger.error("Error processing result from terminating external process", e);
                        }
                    }

                    @Override
                    public void onError(Ack ack) {
                        DesktopUtils.errorAck(listener, ack);
                    }
                };
            }
            this.connection.sendAction("terminate-external-process", terminateExternalProcessPayload, mainCallback, this);
        } catch (Exception e) {
            logger.error("Error termiating external process", e);
            throw new DesktopException(e);
        }
    }


    /**
     *
     * Removes the process entry for the passed UUID obtained
     * from a previous call to DesktopSystem.launchExternalProcess().
     *
     * @param processUuid The UUID for a process launched by DesktopSystem.launchExternalProcess()
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AckListener
     */
    public void releaseExternalProcess(String processUuid, AckListener callback) throws DesktopException{
        try {
            JSONObject releaseExternalProcessPayload = new JSONObject();
            releaseExternalProcessPayload.put("uuid", processUuid);
            this.connection.sendAction("release-external-process", releaseExternalProcessPayload, callback, this);
        } catch (Exception e) {
            logger.error("Error releasing external process", e);
            throw new DesktopException(e);
        }
    }

    /**
     *
     * Copies text to the clipboard
     *
     * @param text The text to copy
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AckListener
     */
    public void setClipboard(String text, AckListener callback) throws DesktopException{
        try {
            JSONObject clipboardPayload = new JSONObject();
            clipboardPayload.put("data", text);
            this.connection.sendAction("set-clipboard", clipboardPayload, callback, this);
        } catch (Exception e) {
            logger.error("Error setting clipboard", e);
            throw new DesktopException(e);
        }
    }

    /**
     * Stores a cookie in the runtime
     *
     * @param url The URL that the cookie is for
     * @param name The key used to lookup the value
     * @param value The value paired with the key (name)
     * @param ttl The time to till the cookie expires in milliseconds.  Never expires when set to 0.  Defaults to 0.
     * @param secure Accessible only on a secured connection (SSL)
     * @param httpOnly Accessible only on HTTP/HTTPS.
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AckListener
     */
    public void setCookie(String url, String name, String value, long ttl, boolean secure, boolean httpOnly, AckListener callback) throws DesktopException{

        try {
            JSONObject setCookiePayload = new JSONObject();
            setCookiePayload.put("url", url);
            setCookiePayload.put("name", name);
            setCookiePayload.put("value", value);
            setCookiePayload.put("ttl", ttl);
            setCookiePayload.put("secure", secure);
            setCookiePayload.put("httpOnly", httpOnly);
            this.connection.sendAction("set-cookie", setCookiePayload, callback, this);
        } catch (Exception e) {
            logger.error("Error publishing cookie", e);
            throw new DesktopException(e);
        }
    }

    /**
     * Returns information about the running RVM
     *
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AckListener
     */
    public void getRvmInfo(AckListener callback) throws DesktopException {
        try {
            this.connection.sendAction("get-rvm-info", null, callback, this);
        } catch (Exception e) {
            logger.error("Error setting env variable", e);
            throw new DesktopException(e);
        }
    }

    /**
     * Returns the version of the runtime
     *
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AckListener
     */
    public void getRuntimeInfo(AckListener callback) throws DesktopException {
        try {
            this.connection.sendAction("get-runtime-info", null, callback, this);
        } catch (Exception e) {
            logger.error("Error setting env variable", e);
            throw new DesktopException(e);
        }
    }

    /**
     * Retrieve name of a environment variable
     *
     * @param name name of the environment variable
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to send the request to Runtime
     * @see AckListener
     * @see DesktopException
     */
    public void getEnvironmentVariable(String name, AckListener callback) throws DesktopException {
        getEnvironmentVariables(new String[]{name}, callback);
    }

    /**
     * Retrieve name of environment variables
     *
     * @param names names of the environment variable
     * @param callback AckListener for the request
     * @throws  DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AckListener
     */
    public void getEnvironmentVariables(String[] names, AckListener callback) throws DesktopException{
        try {
            JSONArray array = new JSONArray();
            for (int i = 0; i < names.length; i++) {
                array.put(names[i]);
            }
            JSONObject payload = new JSONObject();
            payload.put("environmentVariables", array);

            this.connection.sendAction("get-environment-variable", payload, callback, this);
        } catch (Exception e) {
            logger.error("Error setting env variable", e);
            throw new DesktopException(e);
        }
    }

    /**
     * Get system information
     *
     * @param listener AckListener for the request
     * @throws  DesktopException if this method fails to send the request to Runtime
     * @see DesktopException
     * @see AckListener
     */
    public void getHostSpecs(AckListener listener) throws DesktopException {
        try {
            this.connection.sendAction("get-host-specs", null, listener, this);
        } catch (Exception e) {
            logger.error("Error setting env variable", e);
            throw new DesktopException(e);
        }
    }

    /**
     * Get version info of java adapter
     *
     * @return version information
     */
    public static String getAdapterVersion() {
        return adapterVersion;
    }

    /**
     * Get build time of java adapter
     *
     * @return build time
     */
    public static String getAdapterBuildTime() {
        if (adapterBuildTime > 0) {
            SimpleDateFormat fmt = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");
            fmt.setTimeZone(TimeZone.getTimeZone("GMT"));
            return fmt.format(new Date(adapterBuildTime));
        } else {
            return null;
        }
    }


    /**
     * Retrieve info about RVM and Runtime already installed on desktop.  Data is returned as JSONObject
     *
     * Example:
     * {
     * 	"rvm": {
     * 		"path": "C:\\Users\\123\\AppData\\Local\\OpenFin\\OpenFinRVM.exe",
     * 		"version": "5.0.0.9"
     *        },
     * 	"runtime": [{
     * 		"path": "C:\\Users\\123\\AppData\\Local\\OpenFin\\runtime\\10.66.39.43",
     * 		"version": "10.66.39.43"
     *    }, {
     * 		"path": "C:\\Users\\123\\AppData\\Local\\OpenFin\\runtime\\9.61.38.43",
     * 		"version": "9.61.38.43"
     *    }]
     * }
     *
     *
     * @return JSONObject
     */
    public static JSONObject getInstallInfo() {
        return InstallChecker.getInstallInfo();
    }
}
