package com.openfin.desktop;

import java.util.concurrent.CompletionStage;

import org.json.JSONObject;

import com.openfin.desktop.platform.Platform;

/**
 * Layouts give app providers the ability to embed multiple views in a single
 * window. The Layout enables the initialization and manipulation of a window's
 * Layout.
 * 
 * @author Anthony
 *
 */
public class Layout {

	private Identity identity;
	private DesktopConnection connection;
	private Platform platform;

	private Layout(Identity identity, DesktopConnection connection) {
		this.identity = identity;
		this.connection = connection;
		this.platform = Platform.wrap(this.identity.getUuid(), this.connection);
	}

	/**
	 * Returns a Layout object that represents a Window's layout.
	 * @param identity window identity
	 * @param connection Connection object to the AppDesktop.
	 * @return Layout object that represents a Window's layout.
	 */
	public static Layout wrap(Identity identity, DesktopConnection connection) {
		return new Layout(identity, connection);
	}

	/**
	 * Returns the configuration of the window's layout. Returns the same information that is returned for all windows in getSnapshot.
	 * @return New CompletionStage for the configuration of the window's layout.
	 */
	public CompletionStage<LayoutOptions> getConfig() {
		return this.platform.getChannelClient().thenComposeAsync(client -> {
			JSONObject payload = new JSONObject();
			payload.put("target", this.identity.getJson());
			return client.dispatchAsync("get-frame-snapshot", payload).thenApplyAsync(ack -> {
				JSONObject result = ack.getJsonObject().getJSONObject("data").getJSONObject("result");
				return new LayoutOptions(result);
			});
		});
	}

	/**
	 * Replaces a Platform window's layout with a preset layout arrangement using the existing Views attached to the window. The preset options are "columns", "grid", "rows", or "tabs".
	 * @param presetType "columns", "grid", "rows", or "tabs".
	 * @return New CompletionStage when the command is delivered.
	 */
	public CompletionStage<Void> applyPreset(String presetType) {
		return this.platform.getChannelClient().thenComposeAsync(client -> {
			JSONObject presetTypeJson = new JSONObject();
			presetTypeJson.put("presetType", presetType);

			JSONObject payload = new JSONObject();
			payload.put("target", this.identity.getJson());
			payload.put("opts", presetTypeJson);

			return client.dispatchAsync("apply-preset-layout", payload).thenAcceptAsync(ack -> {
				if (!ack.isSuccessful()) {
					throw new RuntimeException("error applyPreset, reason: " + ack.getReason());
				}
			});
		});
	}

	/**
	 * Replaces a Platform window's layout with a new layout. Any views that were in the old layout but not the new layout will be destroyed.
	 * @param newLayout New layout to implement in the target window. 
	 * @return New CompletionStage when the command is delivered.
	 */
	public CompletionStage<Void> replace(LayoutOptions newLayout) {
		return this.platform.getChannelClient().thenComposeAsync(client -> {
			JSONObject layoutJson = new JSONObject();
			layoutJson.put("layout", newLayout.getJson());

			JSONObject payload = new JSONObject();
			payload.put("target", this.identity.getJson());
			payload.put("opts", layoutJson);

			return client.dispatchAsync("replace-layout", payload).thenAcceptAsync(ack -> {
				if (!ack.isSuccessful()) {
					throw new RuntimeException("error replace, reason: " + ack.getReason());
				}
			});
		});
	}
}
