package com.openfin.desktop;

import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.EventObject;

/**
 *  A message that is delivered to an AckListener when an action has
 *  been processed by AppDesktop
 *  @see AckListener
 */
public class Ack extends EventObject {
    private final static Logger logger = LoggerFactory.getLogger(Ack.class.getName());
    private JSONObject object;

    /**
     * Ack constructor
     * @param source The message source
     */
    private Ack(Object source) {
        super(source);
    }

    /**
     * Ack constructor
     * @param object Message being delivered to the listener
     * @param source The message source
     */
    public Ack(JSONObject object, Object source) {
        this(source);
        this.object = object;
    }

    /**
     * Returns Returns true if message contains "success":"true"
     * @return Returns true if message contains "success":"true"
     */
    public boolean isSuccessful() {
        try {
            if (object.has("success"))
                return object.getBoolean("success");
        } catch (JSONException e) {
            logger.error("Error checking success", e);
        }
        return false;
    }

    /**
     * Return reason if Ack has error
     * @return reason for error
     */
    public String getReason() {
        try {
            if (object.has("reason"))
                return object.getString("reason");
        } catch (JSONException e) {
            logger.error("Error checking reason");
        }
        return null;
    }

    /**
     * Returns the message as a JObject
     * @return the message as a JObject
     */
    public JSONObject getJsonObject() {
        return object;
    }

    /**
     * Returns the value of "data" from message as JObject
     * @return the value of "data" from message as JObject
     */
    public Object getData() {
        try {
            if (object.has("data"))
                return object.get("data");
        } catch (JSONException e) {
            logger.error("Error getting data");
        }
        return null;
    }
    
    
    /**
     * When the ack was not successful, it might cause Openfin Desktop Runtime error. This method returns the stack trace of the error, if it's available.
     * @return the stack trace of runtime error, null if not available.
     */
	public String getErrorStack() {
		String errorStack = null;
		try {
			if (object.has("error")) {
				JSONObject errorObj = object.getJSONObject("error");
				errorStack = errorObj.getString("stack");
			}
		}
		catch (JSONException e) {
			logger.error("Error getting error stack");
		}
		return errorStack;
	}

}
