package com.openfin.desktop;

import java.io.*;
import java.net.URISyntaxException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipInputStream;

/**
 * A Description.
 * @author wche
 * @since 2/6/13
 */
public class WebStartInstaller {
    private final static Logger logger = LoggerFactory.getLogger(WebStartInstaller.class.getName());

    private static String desktopPath = "OpenfinDesktop";
    private static String javaDemoPath = "OpenfinDesktopJavaDemo";
    private static String versionProp = "WSDesktopVersion";
    private static String versionFile = "WSDesktopVersion.txt";

    public static void main(final String[] args)
        throws URISyntaxException,
            ZipException,
            IOException
    {
        try {
            if (checkNewVersion()) {
                extractZip("desktop.zip", desktopPath);
                extractZip("java-adapter.zip", javaDemoPath);
                updateExistingVersion();
            }
//            runDesktop();
            runJavaDemo();
        } catch (Exception e) {
            logger.error("Exception in main", e);
        }
        java.lang.System.exit(0);
    }

    private static boolean checkNewVersion() {
        boolean hasNewVersion = true;
        String desktopVersion = java.lang.System.getProperty(versionProp);
        if (desktopVersion != null) {
            logger.debug("checking desktop version " + desktopVersion);
            String  existingVersion = getExistingVersion();
            if (existingVersion != null) {
                hasNewVersion = !desktopVersion.equals(existingVersion);
            }
        }
        if (hasNewVersion) {
            logger.debug("need new desktop version, clearing cached version ");
            clearDirectory(desktopPath);
            clearDirectory(javaDemoPath);
        }
        return hasNewVersion;
    }

    private static String getExistingVersion() {
        String version = null;
        try {
            File vfile = new File(desktopPath + File.separator + versionFile);
            if (vfile.exists()) {
                BufferedReader reader = new BufferedReader(new FileReader(vfile));
                version = reader.readLine();
                reader.close();
            }
        } catch (Exception e) {
            logger.debug("Exception in getExistingVersion", e);
        }
        logger.debug("found cached desktop version " + version);
        return version;
    }

    private static void updateExistingVersion() {
        String desktopVersion = java.lang.System.getProperty(versionProp);
        if (desktopVersion != null) {
            try {
                File vfile = new File(desktopPath + File.separator + versionFile);
                BufferedWriter writer = new BufferedWriter(new FileWriter(vfile, false));
                writer.write(desktopVersion);
                writer.close();
            } catch (Exception e) {
                logger.debug("Exception in updateExistingVersion", e);
            }
            logger.debug("set existing desktop version " + desktopVersion );
        }
    }

    private static void clearDirectory(String directory) {
        try {
            logger.debug("deleting " + directory);
            File dir = new File(directory);
            if (dir.exists()) {
                boolean deleted = dir.delete();
                logger.debug("deleted " + directory + " result " + deleted);
            }
        } catch (Exception e) {
            logger.debug("Exception in clearDirectory", e);
        }
    }

    private static void extractZip(String zipName, String path) {
        try {
            logger.debug("loading resource " + zipName);
            InputStream in = WebStartInstaller.class.getClassLoader().getResourceAsStream(zipName);
            if (in != null) {
                ZipInputStream zipFile = new ZipInputStream(in);
                decompressFile(zipFile, path);
                in.close();
            } else {
                logger.debug("resource desktop.zip missing ");
            }
        } catch (Exception e) {
            logger.error("Exception in extractZip", e);
        }

    }

    private static void decompressFile(ZipInputStream zipFileStream, String destDirectory) throws IOException {
        File destDir = new File(destDirectory);
        if (!destDir.exists()) {
            logger.debug("creating dir " + destDirectory);
            destDir.mkdir();
        }

        ZipEntry entry = zipFileStream.getNextEntry();
        while (entry != null) {
            String filePath = destDirectory + File.separator + entry.getName();
            if (!entry.isDirectory()) {
                File file = new File(filePath);
                createDir(file.getParent());
                extractFile(zipFileStream, filePath);
            } else {
                createDir(filePath);
            }
            zipFileStream.closeEntry();
            entry = zipFileStream.getNextEntry();
        }
        zipFileStream.close();
    }

    private static void createDir(String path) {
        File dir = new File(path);
        if (!dir.exists()) {
            logger.debug("creating dir " + path);
            dir.mkdirs();
        }
    }

    private static void extractFile(ZipInputStream zipIn, String filePath) throws IOException {
        logger.debug("extracting " + filePath);
        BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(filePath));
        byte[] bytesIn = new byte[2048];
        int read = 0;
        while ((read = zipIn.read(bytesIn)) != -1) {
            bos.write(bytesIn, 0, read);
        }
        bos.close();
    }    

    private static void runDesktop() throws Exception {
        logger.debug("starting openfin.exe ");
        String line;
        OutputStream stdin = null;
        InputStream stderr = null;
        InputStream stdout = null;

        Process process = Runtime.getRuntime().exec("cmd");
        stdin = process.getOutputStream();
        stderr = process.getErrorStream();
        stdout = process.getInputStream();

        // "write" the parms into stdin
        line = "cd /d " + desktopPath + "\n";
        stdin.write(line.getBytes());
        stdin.flush();

        line = "start openfin.exe \n";
        stdin.write(line.getBytes());
        stdin.flush();
        stdin.close();

        // clean up if any output in stdout
        BufferedReader brCleanUp =
                new BufferedReader(new InputStreamReader(stdout));
        while ((line = brCleanUp.readLine()) != null) {
            logger.debug("[Stdout] " + line);
        }
        brCleanUp.close();

        // clean up if any output in stderr
        brCleanUp =
                new BufferedReader(new InputStreamReader(stderr));
        while ((line = brCleanUp.readLine()) != null) {
            logger.info("[Stderr] " + line);
        }
        brCleanUp.close();
    }


    private static void runJavaDemo() throws Exception {
        logger.debug("starting java demo ");
        String line;
        OutputStream stdin = null;
        InputStream stderr = null;
        InputStream stdout = null;

        Process process = Runtime.getRuntime().exec("cmd");
        stdin = process.getOutputStream();
        stderr = process.getErrorStream();
        stdout = process.getInputStream();

        // "write" the parms into stdin
        line = "cd /d " + javaDemoPath + "\n";
        logger.debug("cmd: " + line);
        stdin.write(line.getBytes());
        stdin.flush();

        line = "java -cp openfin-desktop-java-adapter.jar com.openfin.desktop.demo.OpenFinDesktopDemo .." +
                File.separator + desktopPath + " 9696 \n ";
        logger.debug("cmd: " + line);

        stdin.write(line.getBytes());
        stdin.flush();
        stdin.close();

        // clean up if any output in stdout
        BufferedReader brCleanUp =
                new BufferedReader(new InputStreamReader(stdout));
        while ((line = brCleanUp.readLine()) != null) {
            logger.debug("[Stdout] " + line);
        }
        brCleanUp.close();

        // clean up if any output in stderr
        brCleanUp =
                new BufferedReader(new InputStreamReader(stderr));
        while ((line = brCleanUp.readLine()) != null) {
            logger.info("[Stderr] " + line);
        }
        brCleanUp.close();
    }

}
