package com.openfin.desktop;

import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Helper object that provides getters/setters for the
 * various options needed for creating an Application.
 */
public class ApplicationOptions {
    private final static Logger logger = LoggerFactory.getLogger(ApplicationOptions.class.getName());

    JSONObject options;
    WindowOptions mainWindowOptions;

    /**
     * Constructs an instance with the passed options.
     * @param options Options for the application
     */
    public ApplicationOptions(JSONObject options) {
        this.options = options;
    }

    /**
     * Constructor
     * @param name Name of the application
     * @param uuid UUID of the application
     * @param url  URL of the application
     */
    public ApplicationOptions(String name, String uuid, String url) {
        options = new JSONObject();
        try {
            options.put("name", name);
            options.put("uuid", uuid);
            options.put("url", url);
        } catch (JSONException e) {
            logger.error("Error populating options", e);
        }
    }

    /**
     * Gets name of the application
     * @return Name of the application
     */
    public String getName() {
        return getStringValue("name");
    }

    /**
     * Gets UUID of the application
     * @return UUID of the application
     */
    public String getUUID() {
        return getStringValue("uuid");
    }

    /**
     * Gets URL of the application
     * @return URL of the application
     */
    public String getURL() {
        return getStringValue("url");
    }

    /**
     * Sets options of main window of the application
     * @param options Options of main window
     */
    public void setMainWindowOptions(WindowOptions options) {
        mainWindowOptions = options;
    }

    /**
     * Get options of main window
     * @return Options of main window
     */
    public WindowOptions getMainWindowOptions() {
        return mainWindowOptions;
    }

    /**
     * Sets URL of application icon
     * @param applicationIcon URL
     */
    public void setApplicationIcon(String applicationIcon) {
        try {
            options.put("applicationIcon", applicationIcon);
        } catch (JSONException e) {
            logger.error("Error setting applicationIcon");
        }
    }

    /**
     * Gets URL of application icon
     * @return URL
     */
    public String getApplicationIcon() {
        return getStringValue("applicationIcon");
    }

    /**
     * Sets version of the application
     * @param version Version
     */
    public void setVersion(String version) {
        try {
            options.put("version", version);
        } catch (JSONException e) {
            logger.error("Error setting version");
        }
    }

    /**
     * Gets version of the application
     * @return  Version
     */
    public String getVersion() {
        return getStringValue("version");
    }

    /**
     * Gets JSON representation of options
     * @return JSONObject
     */
    public JSONObject getJson() {
        try {
            this.options.put("mainWindowOptions", this.mainWindowOptions.getJson());
            return new JSONObject(options.toString());
        } catch (JSONException e) {
            logger.error("Error creating JSON object", e );
        }
        return null;
    }

    /**
     * Helper method for retrieving string value of a property
     * @param property Property name
     * @return Value of the property
     */
    protected String getStringValue(String property) {
        String value = null;
        try {
            value =  options.getString(property);
        } catch (JSONException e) {
        }
        return value;
    }

    /**
     * Set value for a property
     * @param key name of the property
     * @param value value
     * @return this
     * @throws JSONException from JSONObject.put
     */
     public ApplicationOptions put(String key, Object value) throws JSONException {
        this.options.put(key, value);
        return this;
     }
}
