package com.openfin.desktop;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Configuration for launching OpenFin Runtime
 *
 * Created by wche on 3/26/2016.
 */
public class RuntimeConfiguration extends JSONObject {

    private int devToolsPort;
    private String runtimeVersion;
    private String securityRealm;
    private String additionalRuntimeArguments;
    private JSONObject startupApp;
    private List<JSONObject> appAssets;
    private String rdmURL;
    private String runtimeAssetURL;
    private String additionalRvmArguments;
    private Map<String, Object> configMap = new HashMap<String, Object>();  // used to add any random config setting to app json file
    private boolean launchWithRVM = false;  // true -> launch RVM instead of installer

    /**
     * Constructor
     */
    public RuntimeConfiguration() {
        this.appAssets = new ArrayList<JSONObject>();
    }

    /**
     * Set port for accessing devtools on localhost
     * @param port port number
     */
    public void setDevToolsPort(int port) {
        this.devToolsPort = port;
    }

    /**
     * Set version number of Runtime to launch
     * @param version version number
     */
    public void setRuntimeVersion(String version) {
        this.runtimeVersion = version;
    }

    /**
     * Get version number of Runtime
     * @return version of Runtime
     */
    public String getRuntimeVersion() {
        return this.runtimeVersion;
    }

    /**
     * Set security realm of Runtime
     *
     * @param securityRealm name of security realm
     */
    public void setSecurityRealm(String securityRealm) {
        this.securityRealm = securityRealm;
    }

    /**
     * Set additional arguments for Runtime
     *
     * @param additionalRuntimeArguments additional arguments
     */
    public void setAdditionalRuntimeArguments(String additionalRuntimeArguments) {
        this.additionalRuntimeArguments = additionalRuntimeArguments;
    }

    /**
     * Add configuration of an app asset
     *
     * @param assetConfig configuration in JSON format
     */
    public void addAppAsset(JSONObject assetConfig) {
        this.appAssets.add(assetConfig);
    }

    /**
     * Set URL of RDM service
     *
     * @param rdmURL RDM URL
     */
    public void setRdmURL(String rdmURL) {
        this.rdmURL = rdmURL;
    }

    /**
     * Get URL of RDM service
     *
     * @return URL of RDM service
     */
    public String getRdmURL() {
        return this.rdmURL;
    }

    /**
     * Set URL of Runtime assets, including RVM and Runtime
     *
     * @param runtimeAssetURL URL of Runtime assets
     */
    public void setRuntimeAssetURL(String runtimeAssetURL) {
        this.runtimeAssetURL = runtimeAssetURL;
    }

    /**
     * Get URL of Runtime assets
     *
     * @return URL of Runtime assets
     */
    public String getRuntimeAssetURL() {
        return this.runtimeAssetURL;
    }

    /**
     * Get additional arguments for RVM and Installer
     *
     * @return additional arguments
     */
    public String getAdditionalRvmArguments() {
        return additionalRvmArguments;
    }

    /**
     * set Defines the minimum RVM version the app requires. This option is available in RVM 2.3 or greater
     *
     * @param rvmVersion version of RVM required
     */
    public void setRvmVersion(String rvmVersion) {
        this.addConfigurationItem("rvmVersion", rvmVersion);
    }

    /**
     * set configuration for shortcut
      * @param shortCut config for shortcut
     */
    public void setShortCut(JSONObject shortCut) {
        this.addConfigurationItem("shortcut", shortCut);
    }

    /**
     * set configuration to customize the appearance of the RVM progress dialog
     *
     * @param dialogSettings
     */
    public void setDialogSettings(JSONObject dialogSettings) {
        this.addConfigurationItem("dialogSettings", dialogSettings);
    }

    /**
     * Specify an image to display while the runtime is loading. It takes any image file (including semi-transparent PNGs)
     *
     * @param splashScreenImage URL of the image
     */
    public void setSplashScreenImage(String splashScreenImage) {
        this.addConfigurationItem("splashScreenImage", splashScreenImage);
    }

    /**
     * Specify customized error messages during launch
     *
     * @param supportInformation config for support info
     */
    public void setSupportInformation(String supportInformation) {
        this.addConfigurationItem("supportInformation", supportInformation);
    }

    /**
     * Set additional arguments for RVM and Installer
     *
     * @param additionalRvmArguments set additional arguments
     */
    public void setAdditionalRvmArguments(String additionalRvmArguments) {
        this.additionalRvmArguments = additionalRvmArguments;
    }

    /**
     * Set configuration of starutp application.
     *
     * @param startupApp configuration in JSON format
     */
    public void setStartupApp(JSONObject startupApp) {
        this.startupApp = startupApp;
    }

    /**
     * Add a configuration setting for launching Runtime.  value can be type of Integer, Long, Double, String or JSONObject
     *
     * @param key key value of the setting
     * @param value value of the setting
     */
    public void addConfigurationItem(String key, Object value) {
        this.configMap.put(key, value);
    }

    /**
     * By default, Java adapter launches Runtime with OpenFin installer.  If launchWithRVM is true, installer is skipped and RVM is started instead.
     *
     * @param launchWithRVM true or false
     */
    public void setLaunchWithRVM(boolean launchWithRVM) {
        this.launchWithRVM = launchWithRVM;
    }

    /**
     * return value of launchWithRVM
     *
     * @return
     */
    public boolean isLaunchWithRVM() {
        return this.launchWithRVM;
    }

    /**
     * Generates config for "runtime" section
     */
    private void appendRuntimeOptions(JSONObject rootConfig) {
        JSONObject runtimeConfig = new JSONObject();
        StringBuffer arguments = new StringBuffer();
        if (this.securityRealm != null) {
            arguments.append(String.format(" --security-realm=%s ", this.securityRealm));
        }
        if (this.additionalRuntimeArguments != null) {
            arguments.append(String.format(" %s ", this.additionalRuntimeArguments));
        }
        runtimeConfig.put("arguments", arguments);
        runtimeConfig.put("version", this.runtimeVersion);
        rootConfig.put("runtime", runtimeConfig);
    }

    /**
     * Append app assets config to a root configuration
     *
     * @param rootConfig root level configuration
     */
    private void appendAppAssets(JSONObject rootConfig) {
        if (this.appAssets.size() > 0) {
            rootConfig.put("appAssets", this.appAssets);
        }
    }

    /**
     * Append startup_app config to a root configuration
     *
     * @param rootConfig root level configuration
     */
    private void appendStartupConfig(JSONObject rootConfig) {
        if (this.startupApp != null) {
            rootConfig.put("startup_app", this.startupApp);
        }
    }

    private void appendAdditionalConfigItems(JSONObject rootConfig) {
        if (this.configMap.size() > 0) {
            for (String key : this.configMap.keySet()) {
                Object value = this.configMap.get(key);
                if (value instanceof Double) {
                    rootConfig.put(key, ((Double) value).doubleValue());
                }
                else if (value instanceof Integer) {
                    rootConfig.put(key, ((Integer) value).intValue());
                }
                else if (value instanceof Long) {
                    rootConfig.put(key, ((Long) value).longValue());
                } else {
                    rootConfig.put(key, value);
                }
            }
        }
    }

    /**
     * Append rdmUrl config to a root configuration
     *
     * @param rootConfig root level configuration
     */
    private void appendRDMURL(JSONObject rootConfig) {
        if (this.rdmURL != null) {
            rootConfig.put("rdmUrl", this.rdmURL);
        }
    }

    /**
     * Append assetsUrl config to a root configuration
     *
     * @param rootConfig root level configuration
     */
    private void appendRuntimeAssetsURL(JSONObject rootConfig) {
        if (this.runtimeAssetURL != null) {
            rootConfig.put("assetsUrl", this.runtimeAssetURL);
        }
    }

    /**
     * Generates JSON string that can be passed as --config to RVM and Insaller
     *
     * @return JSON string
     */
    public String generateRuntimeConfig() {
        JSONObject config = new JSONObject();
        if (this.devToolsPort > 0) {
            config.put("devtools_port", this.devToolsPort);
        }
        appendRuntimeOptions(config);
        appendRDMURL(config);
        appendRuntimeAssetsURL(config);
        appendAppAssets(config);
        appendStartupConfig(config);
        appendAdditionalConfigItems(config);
        return config.toString();
    }


}
