package com.openfin.desktop;

import com.openfin.desktop.*;
import org.json.JSONException;
import org.json.JSONObject;

/**
 *
 * A Notification represents a window on App Desktop which is shown briefly
 * to the user on the bottom-right corner of the primary monitor.
 * A notification is typically used to alert the user of some important event
 * which requires his or her attention.
 *
 *  Multiple notifications can be generated at once but will queue if more
 * than 5 are already displayed. Notifications can be dismissed by dragging
 * them to the right with the mouse and can communicate securely with their
 * invoking applications.
 *
 * @author wche
 * @since 9/4/14
 */
public class Notification {

    private DesktopConnection desktopConnection;
    private Integer id = 0;
    private JSONObject closePayload;
    private JSONObject sendMessagePayload;

    /**
     * Notification constructor
     * @param notificationOptions The options of this notification
     * @param desktopConnection Connection to Desktop
     * @param notificationListener NotificationListener for the notification
     * @see NotificationListener
     * @param listener AckListener for the request
     * @see AckListener
     * @throws Exception if notification fails to be created
     */
    public Notification(NotificationOptions notificationOptions, NotificationListener notificationListener, DesktopConnection desktopConnection, AckListener listener) throws Exception {
        this.desktopConnection = desktopConnection;
        if (notificationOptions != null && notificationListener != null) {
            this.id = desktopConnection.registerNotificationListener(notificationListener, this);
            if(id > 0)  {
                JSONObject payload = new JSONObject();
                JsonUtils.updateValue(payload, "url", notificationOptions.getURL());
                JsonUtils.updateValue(payload, "notificationId", this.id);
                if (notificationOptions.getMessage() != null) {
                    JsonUtils.updateValue(payload, "message", notificationOptions.getMessage());
                } else {
                    JsonUtils.updateValue(payload, "message", notificationOptions.getMessageText());
                }
                if (notificationOptions.getTimeout() > -1) {
                    JsonUtils.updateValue(payload, "timeout", notificationOptions.getTimeout());
                }
                else {
                    JsonUtils.updateValue(payload, "timeout", "never");
                }
                this.desktopConnection.sendActionToNotificationsCenter("create-notification", payload, listener, this);
            }
        }
    }

    /**
     *
     * Closes the notification
     *
     * @param listener AckListener for the request
     * @see AckListener
     */
    public void close(AckListener listener) {
        try {
            if (closePayload == null) {
                closePayload = new JSONObject();
                JsonUtils.updateValue(closePayload, "notificationId", this.id);
            }
            desktopConnection.sendActionToNotificationsCenter("close-notification", closePayload, listener, this);
        } catch (Exception ex) {
            DesktopUtils.errorAckOnException(listener, this, ex);
        }
    }

    /**
     *
     * Sends a message to the notification
     *
     * @param message The JSON message to be sent to the notification
     * @param listener AckListener for the request
     * @see AckListener
     */
    public void sendMessage(JSONObject message, AckListener listener) {
        generateNotificationMessage("send-notification-message", message, listener);
    }

    /**
     *
     * Sends a message to the notification
     *
     * @param message The String message to be sent to the notification
     * @param listener AckListener for the request
     * @see AckListener
     */
    public void sendMessage(String message, AckListener listener) {
        this.generateNotificationMessage("send-notification-message", message, listener);
    }

    /**
     * Generate and send notification message
     *
     * @param action The notification action, send-notification-message or send-application-message
     * @param message The message to be sent to the notification
     * @param listener AckListener for the request
     * @see AckListener
     */
    private void generateNotificationMessage(String action, Object message, AckListener listener) {
        try {
            if (sendMessagePayload == null) {
                sendMessagePayload = new JSONObject();
                JsonUtils.updateValue(sendMessagePayload, "notificationId", this.id);
            }
            if (message instanceof String) {
                JsonUtils.updateValue(sendMessagePayload, "message", (String)message);
            } else {
                JsonUtils.updateValue(sendMessagePayload, "message", (JSONObject)message);
            }
            desktopConnection.sendActionToNotificationsCenter(action, sendMessagePayload, listener, this);
        } catch (Exception ex) {
            DesktopUtils.errorAckOnException(listener, this, ex);
        }
    }

}
