package com.openfin.desktop;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Helper methods for retrieving data from JSONObject
 */

public class JsonUtils {

    /**
     * Retrieve a string value from a JSONObject
     * @param jsonObject JSONObject to get value from
     * @param property Property name of the value
     * @param defaultValue Default value if the property does not exist
     * @return the String value
     */
    public static String getStringValue(JSONObject jsonObject, String property, String defaultValue) {
        String value = null;
        try {
            value =  jsonObject.getString(property);
        } catch (JSONException e) {
        } finally {
            if (value == null) {
                value = defaultValue;
            }
        }
        return value;
    }

    /**
     * Update a String value in an JSONOject
     * @param jsonObject JSONObject to update
     * @param property Name of the property
     * @param value New value
     * @throws JSONException if update fails
     */
    public static void updateValue(JSONObject jsonObject, String property, String value) throws JSONException {
        if (value != null) {
            jsonObject.put(property, value);
        }
    }

    /**
     * Retrieve an Integer value from a JSONObject
     * @param jsonObject JSONObject to get value from
     * @param property Property name of the value
     * @param defaultValue Default value if the property does not exist
     * @return the Integer value
     */
    public static Integer getIntegerValue(JSONObject jsonObject, String property, Integer defaultValue) {
        Integer value = null;
        try {
            value = jsonObject.getInt(property);
        } catch (JSONException e) {
        } finally {
            if (value == null) {
                value = defaultValue;
            }
        }
        return value;
    }

    /**
     * Update a Integer value in an JSONOject
     * @param jsonObject JSONObject to update
     * @param property Name of the property
     * @param value New value
     * @throws JSONException if update fails
     */
    public static void updateValue(JSONObject jsonObject, String property, Integer value) throws JSONException {
        if (value != null) {
            jsonObject.put(property, value);
        }
    }

    /**
     * Retrieve an Double value from a JSONObject
     * @param jsonObject JSONObject to get value from
     * @param property Property name of the value
     * @param defaultValue Default value if the property does not exist
     * @return the Integer value
     */
    public static Double getDoubleValue(JSONObject jsonObject, String property, Double defaultValue) {
        Double value = null;
        try {
            value =  jsonObject.getDouble(property);
        } catch (JSONException e) {
        } finally {
            if (value == null) {
                value = defaultValue;
            }
        }
        return value;
    }

    /**
     * Update a Double value in an JSONOject
     * @param jsonObject JSONObject to update
     * @param property Name of the property
     * @param value New value
     * @throws JSONException if update fails
     */
    public static void updateValue(JSONObject jsonObject, String property, Double value) throws JSONException {
        if (value != null) {
            jsonObject.put(property, value);
        }
    }

    /**
     * Retrieve an Boolean value from a JSONObject
     * @param jsonObject JSONObject to get value from
     * @param property Property name of the value
     * @param defaultValue Default value if the property does not exist
     * @return the Integer value
     */
    public static Boolean getBooleanValue(JSONObject jsonObject, String property, Boolean defaultValue) {
        Boolean value = null;
        try {
            if (value != null) {
                value = jsonObject.getBoolean(property);
            }
        } catch (JSONException e) {
        } finally {
            if (value == null) {
                value = defaultValue;
            }
        }
        return value;
    }

    /**
     * Update a Boolean value in an JSONOject
     * @param jsonObject JSONObject to update
     * @param property Name of the property
     * @param value New value
     * @throws JSONException if update fails
     */
    public static void updateValue(JSONObject jsonObject, String property, Boolean value) throws JSONException {
        if (value != null) {
            jsonObject.put(property, value);
        }
    }

    /**
     * Retrieve an JSONObject value from a JSONObject
     * @param jsonObject JSONObject to get value from
     * @param property Property name of the value
     * @param defaultValue Default value if the property does not exist
     * @return the Integer value
     */
    public static JSONObject getJsonValue(JSONObject jsonObject, String property, JSONObject defaultValue) {
        JSONObject value = null;
        try {
            value =  jsonObject.getJSONObject(property);
        } catch (JSONException e) {
        } finally {
            if (value == null) {
                value = defaultValue;
            }
        }
        return value;
    }


    /**
     * Update a JSONObject value in an JSONOject
     * @param jsonObject to update
     * @param property Name of the property
     * @param value New value
     * @throws JSONException if update fails
     */
    public static void updateValue(JSONObject jsonObject, String property, JSONObject value) throws JSONException {
        if (value != null) {
            jsonObject.put(property, value);
        }
    }

    /**
     * Retrieve an JSONArray value from a JSONObject
     * @param jsonObject JSONObject to get value from
     * @param property Property name of the value
     * @param defaultValue Default value if the property does not exist
     * @return the Integer value
     */
    public static JSONArray getJsonArray(JSONObject jsonObject, String property, JSONArray defaultValue) {
        JSONArray array = null;
        try {
            array = jsonObject.getJSONArray(property);
        } catch (JSONException s) {
        } finally {
            if (array == null) {
                array = defaultValue;
            }
        }
        return array;
    }

    /**
     * Update a JSONArray value in an JSONOject
     * @param jsonObject to update
     * @param property Name of the property
     * @param value New value
     * @throws JSONException if update fails
     */
    public static void updateValue(JSONObject jsonObject, String property, JSONArray value) throws JSONException {
        if (value != null) {
            jsonObject.put(property, value);
        }
    }



}
