package com.openfin.desktop;

import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

/**
 * Helper methods
 *
 * @author wche
 * @since 12/8/14
 */
public class DesktopUtils {
    private final static Logger logger = LoggerFactory.getLogger(DesktopUtils.class.getName());

    /**
     * Call onError of an AckListen for an exception
     *
     * @param ackListener AckListener to be called
     * @param source Source of the Ack
     * @param ex Exception for the Ack
     * @see Ack
     * @see AckListener
     *
     */
    public static void errorAckOnException(AckListener ackListener, Object source, Exception ex) {
        if (ackListener != null) {
            try {
                JSONObject payload = new JSONObject();
                JsonUtils.updateValue(payload, "reason", ex.getMessage());
                StringWriter stringWriter = new StringWriter();
                PrintWriter printWriter = new PrintWriter(stringWriter);
                ex.printStackTrace(printWriter);
                JsonUtils.updateValue(payload, "stacktrace", stringWriter.toString());
                Ack ack = new Ack(payload, source);
                ackListener.onError(ack);
                errorAck(ackListener, ack);
            } catch (Exception e) {
                logger.error("Error", e);
            }
        }
    }

    /**
     * Call onSuccess of an AckListen
     * @param ackListener AckListener to be called
     * @param ack an instance of Ack to be passed to ackListener
     * @see Ack
     * @see AckListener
     */
    public static void successAck(AckListener ackListener, Ack ack) {
        try {
            if (ackListener != null) {
                ackListener.onSuccess(ack);
            }
        } catch (Exception ex) {
            logger.error("Error invoking success callback", ex);
        }
    }
    public static void errorAck(AckListener callback, Ack ack) {
        try {
            if (callback != null) {
                callback.onError(ack);
            }
        } catch (Exception ex) {
            logger.error("Error invoking error callback", ex);
        }
    }

    /**
     * Add event listener to a window.  Since this blocks for response, it should NOT called on websocket read thread.
     *
     * @param window an instance of Window
     * @param evenType type of event
     * @param eventListener EventListner to be added
     * @throws Exception if this method fails to add event listener
     */
    public static void addEventListener(Window window, final String evenType, EventListener eventListener) throws Exception {
        logger.debug("addEventListener " + evenType);
        final CountDownLatch latch = new CountDownLatch(1);
        window.addEventListener(evenType, eventListener, new AckListener() {
            @Override
            public void onSuccess(Ack ack) {
                latch.countDown();
                logger.debug("addEventListener ack " + ack.isSuccessful());
            }
            @Override
            public void onError(Ack ack) {
                logger.error(String.format("Error adding event listener %s %s", evenType, ack.getReason()));
            }
        });
        latch.await(5, TimeUnit.SECONDS);
        if (latch.getCount() != 0) {
            logger.error("Error adding event listener " + evenType);
        }
    }

}
