package cn.zoecloud.core.service;

import cn.zoecloud.core.ClientConfiguration;
import cn.zoecloud.core.ClientException;
import cn.zoecloud.core.ServiceException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.util.EntityUtils;

import javax.net.ssl.SSLContext;
import java.io.IOException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

import static cn.zoecloud.core.util.LogUtil.logException;

/**
 * 服务调用客户端
 * @author Leo
 */
public class ServiceClient {
    protected static HttpRequestFactory httpRequestFactory = new HttpRequestFactory();

    private ClientConfiguration config;

    protected CloseableHttpClient httpClient;
    protected HttpClientConnectionManager connectionManager;
    protected RequestConfig requestConfig;

    /**
     * 根据客户端配置信息实例化服务客户端
     * @param config
     */
    public ServiceClient(ClientConfiguration config) {
        this.config = config;

        this.connectionManager = createHttpClientConnectionManager();
        this.httpClient = createHttpClient(this.connectionManager);
        RequestConfig.Builder requestConfigBuilder = RequestConfig.custom();
        requestConfigBuilder.setConnectTimeout(config.getConnectionTimeout());
        requestConfigBuilder.setSocketTimeout(config.getSocketTimeout());
        requestConfigBuilder.setConnectionRequestTimeout(config.getConnectionRequestTimeout());
        this.requestConfig = requestConfigBuilder.build();
    }

    /**
     * 发送请求并等待响应
     * @param request
     * @return 封装过的响应消息
     */
    public ResponseMessage sendRequest(RequestMessage request)
            throws ServiceException, ClientException {

        assert(request != null);

        try {
            return this.sendRequestCore(request);
        } catch (ServiceException sex) {
            logException("[Server]Unable to execute HTTP request: ", sex);
            throw sex;
        } catch (ClientException cex) {
            logException("[Client]Unable to execute HTTP request: ", cex);
            throw cex;
        } catch (Exception ex) {
            logException("[Unknown]Unable to execute HTTP request: ", ex);
            throw new ClientException(String.format("Connection error due to: %s", ex.getMessage()));
        }
    }

    /**
     * 实际发送请求
     * @param request
     * @return 封装过的响应消息
     * @throws IOException
     */
    public ResponseMessage sendRequestCore(RequestMessage request) throws IOException {
        HttpRequestBase httpRequest = httpRequestFactory.createHttpRequest(request);

        CloseableHttpResponse httpResponse = null;
        try {
            httpResponse = httpClient.execute(httpRequest);
        } catch (IOException ex) {
            httpRequest.abort();
            throw new ClientException("Network exception", ex);
        }

        return buildResponse(httpResponse);
    }

    public void shutdown() {
        IdleConnectionReaper.removeConnectionManager(this.connectionManager);
        this.connectionManager.shutdown();
    }

    /**
     * 创建CloseableHttpClient
     * @param connectionManager
     * @return
     */
    private CloseableHttpClient createHttpClient(HttpClientConnectionManager connectionManager) {
        return HttpClients.custom()
                .setConnectionManager(connectionManager)
                .setUserAgent(this.config.getUserAgent())
                .disableContentCompression()
                .build();
    }

    /**
     * 创建HttpClientConnectionManager
     * @return
     */
    private HttpClientConnectionManager createHttpClientConnectionManager() {
        SSLContext sslContext = null;
        try {
            sslContext = new SSLContextBuilder().loadTrustMaterial(null, new TrustStrategy() {
                @Override
                public boolean isTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                    return true;
                }
            }).build();

        } catch (Exception e) {
            throw new ClientException(e.getMessage());
        }

        SSLConnectionSocketFactory sslSocketFactory = new SSLConnectionSocketFactory(sslContext,
                NoopHostnameVerifier.INSTANCE);
        Registry<ConnectionSocketFactory> socketFactoryRegistry = RegistryBuilder.<ConnectionSocketFactory> create()
                .register(Protocol.HTTP.toString(), PlainConnectionSocketFactory.getSocketFactory())
                .register(Protocol.HTTPS.toString(), sslSocketFactory).build();

        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(socketFactoryRegistry);
        connectionManager.setDefaultMaxPerRoute(config.getMaxConnections());
        connectionManager.setMaxTotal(config.getMaxConnections());
        connectionManager.setValidateAfterInactivity(config.getValidateAfterInactivity());
        connectionManager.setDefaultSocketConfig(
                SocketConfig.custom().setSoTimeout(config.getSocketTimeout()).setTcpNoDelay(true).build());

        IdleConnectionReaper.setIdleConnectionTime(config.getIdleConnectionTime());
        IdleConnectionReaper.registerConnectionManager(connectionManager);

        return connectionManager;
    }

    /**
     * 创建响应消息
     * @param httpResponse
     * @return
     * @throws IOException
     */
    private static ResponseMessage buildResponse(CloseableHttpResponse httpResponse)
            throws IOException {

        assert(httpResponse != null);

        ResponseMessage response = new ResponseMessage();

        if (httpResponse.getStatusLine() != null) {
            response.setStatusCode(httpResponse.getStatusLine().getStatusCode());
        }

        if (httpResponse.getEntity() != null) {
            response.setContent(EntityUtils.toString(httpResponse.getEntity()));
        }

        return response;
    }
}
