package cn.zoecloud.core.internal;

import cn.zoecloud.core.ServiceException;
import cn.zoecloud.core.model.*;
import cn.zoecloud.core.model.comm.GenericPageResult;
import cn.zoecloud.core.model.comm.ZoeCloudResponse;
import cn.zoecloud.core.parser.ResponseParseException;
import cn.zoecloud.core.parser.ResponseParser;
import com.alibaba.fastjson.JSONObject;

import java.util.ArrayList;
import java.util.List;

import static cn.zoecloud.core.model.Command.*;

/**
 * 解析器集合
 * @author Leo
 */
public class ZoeCloudResponseParsers {
    public static final GetGroupPageResponseParser getGroupPageResponseParser = new GetGroupPageResponseParser();
    public static final GetTerminalSingleResponseParser getTerminalSingleResponseParser = new GetTerminalSingleResponseParser();
    public static final GetTerminalPageResponseParser getTerminalPageResponseParser = new GetTerminalPageResponseParser();
    public static final CommandResultListResponseParser commandResultListResponseParser = new CommandResultListResponseParser();
    public static final GetLayoutPageResponseParser getLayoutPageResponseParser = new GetLayoutPageResponseParser();
    public static final ProgramPublishResponseParser programPublishResponseParser = new ProgramPublishResponseParser();
    public static final GetProgramPublishStatusResponseParser getProgramPublishStatusResponseParser = new GetProgramPublishStatusResponseParser();
    public static final GetProgramPublishDetailResponseParser getProgramPublishDetailResponseParser = new GetProgramPublishDetailResponseParser();

    private static final String NODE_PAGE_CONTENT = "content";
    private static final String NODE_PAGE_NUMBER = "number";
    private static final String NODE_PAGE_SIZE = "size";
    private static final String NODE_PAGE_TOTAL_ELEMENTS = "totalElements";

    public static final class GetGroupPageResponseParser implements ResponseParser<GetGroupPageResult> {
        @Override
        public GetGroupPageResult parse(String responseBody) throws ResponseParseException {
            ZoeCloudResponse response = parseResponseString(responseBody);
            GenericPageResult genericPageResult = parsePageInfo(response);

            GetGroupPageResult result = new GetGroupPageResult(genericPageResult);
            List<Group> groups = response.getPage().getJSONArray(NODE_PAGE_CONTENT).toJavaList(Group.class);
            result.setEntities(groups);
            return result;
        }
    }

    public static final class GetTerminalSingleResponseParser implements ResponseParser<GetTerminalSingleResult> {
        @Override
        public GetTerminalSingleResult parse(String responseBody) throws ResponseParseException {
            GetTerminalSingleResult result = new GetTerminalSingleResult();
            ZoeCloudResponse response = parseResponseString(responseBody);
            if (response.getEntity() != null) {
                result.setEntity(response.getEntity().toJavaObject(Terminal.class));
            }
            return result;
        }
    }

    public static final class GetTerminalPageResponseParser implements ResponseParser<GetTerminalPageResult> {
        @Override
        public GetTerminalPageResult parse(String responseBody) throws ResponseParseException {
            ZoeCloudResponse response = parseResponseString(responseBody);
            GenericPageResult genericPageResult = parsePageInfo(response);

            GetTerminalPageResult result = new GetTerminalPageResult(genericPageResult);
            List<Terminal> terminals = response.getPage().getJSONArray(NODE_PAGE_CONTENT).toJavaList(Terminal.class);
            result.setEntities(terminals);
            return result;
        }
    }

    public static final class CommandResultListResponseParser implements ResponseParser<CommandResultListResult> {
        @Override
        public CommandResultListResult parse(String responseBody) throws ResponseParseException {
            ZoeCloudResponse response = parseResponseString(responseBody);
            if (response.getEntities() == null || response.getEntities().size() == 0) {
                throw new ResponseParseException("The result of commands is empty.");
            }

            List<CommandResult> entities = new ArrayList<>();
            for (int i = 0, len = response.getEntities().size(); i < len; i++) {
                JSONObject json = response.getEntities().getJSONObject(i);
                entities.add(parserCommandResult(json));
            }

            CommandResultListResult result = new CommandResultListResult();
            result.setEntities(entities);
            return result;
        }
    }

    public static final class GetLayoutPageResponseParser implements ResponseParser<GetLayoutPageResult> {
        @Override
        public GetLayoutPageResult parse(String responseBody) throws ResponseParseException {
            ZoeCloudResponse response = parseResponseString(responseBody);
            GenericPageResult genericPageResult = parsePageInfo(response);

            GetLayoutPageResult result = new GetLayoutPageResult(genericPageResult);
            List<Layout> layouts = response.getPage().getJSONArray(NODE_PAGE_CONTENT).toJavaList(Layout.class);
            result.setEntities(layouts);
            return result;
        }
    }

    public static final class ProgramPublishResponseParser implements ResponseParser<ProgramPublishResult> {
        @Override
        public ProgramPublishResult parse(String responseBody) throws ResponseParseException {
            ProgramPublishResult result = new ProgramPublishResult();
            ZoeCloudResponse response = parseResponseString(responseBody);
            if (response.getEntity() != null) {
                result.setEntity(response.getEntity().toJavaObject(PublishInfo.class));
            }
            return result;
        }
    }

    public static final class GetProgramPublishStatusResponseParser implements ResponseParser<GetProgramPublishStatusResult> {
        @Override
        public GetProgramPublishStatusResult parse(String responseBody) throws ResponseParseException {
            GetProgramPublishStatusResult result = new GetProgramPublishStatusResult();
            ZoeCloudResponse response = parseResponseString(responseBody);
            if (response.getEntity() != null) {
                result.setEntity(response.getEntity().toJavaObject(PublishStatusInfo.class));
            }
            return result;
        }
    }

    public static final class GetProgramPublishDetailResponseParser implements ResponseParser<GetProgramPublishDetailResult> {
        @Override
        public GetProgramPublishDetailResult parse(String responseBody) throws ResponseParseException {
            ZoeCloudResponse response = parseResponseString(responseBody);
            if (response.getEntities() == null || response.getEntities().size() == 0) {
                throw new ResponseParseException("The result is empty.");
            }

            GetProgramPublishDetailResult result = new GetProgramPublishDetailResult();
            result.setEntities(response.getEntities().toJavaList(PublishDetail.class));
            return result;
        }
    }

    /**
     * 解析字符串到响应的数据结构
     * @param responseBody
     * @return
     * @throws ResponseParseException
     */
    private static ZoeCloudResponse parseResponseString(String responseBody) throws ResponseParseException {
        ZoeCloudResponse response = JSONObject.parseObject(responseBody, ZoeCloudResponse.class);
        if (response == null) {
            throw new ResponseParseException("Unable to parse the response content.");
        }
        if (!response.isSuccess()) {
            throw new ServiceException(response.getMessage(), response.getErrorCode(), responseBody);
        }
        return response;
    }

    /**
     * 解析分页数据
     * @param response
     * @return
     * @throws ResponseParseException
     */
    private static GenericPageResult parsePageInfo(ZoeCloudResponse response) throws ResponseParseException {
        JSONObject page = response.getPage();
        if (page == null) {
            throw new ResponseParseException("Node page cannot be found.");
        }
        Integer pageNumber = page.getInteger(NODE_PAGE_NUMBER);
        Integer pageSize = page.getInteger(NODE_PAGE_SIZE);
        Integer totalElements = page.getInteger(NODE_PAGE_TOTAL_ELEMENTS);
        if (pageNumber == null || pageSize == null || totalElements == null) {
            throw new ResponseParseException("Paging node data exception");
        }

        GenericPageResult genericPageResult = new GenericPageResult();
        genericPageResult.setPageNumber(pageNumber);
        genericPageResult.setPageSize(pageSize);
        genericPageResult.setTotalElements(totalElements);

        return genericPageResult;
    }

    /**
     * 解析命令结果
     * @param json
     * @return
     * @throws ResponseParseException
     */
    private static CommandResult parserCommandResult(JSONObject json) throws ResponseParseException {
        Command command = findOne(json.getString("command"));
        if (command == null) {
            throw new ResponseParseException("Command not found.");
        }
        CommandStatus status = CommandStatus.findOne(json.getString("status"));
        if (status == null) {
            throw new ResponseParseException("Command status exception");
        }

        CommandResult commandResult = null;
        if (status == CommandStatus.SUCCESSFUL) {
            switch (command) {
                case SCREEN_STATUS:
                case SCREEN_CAPTURE:
                    commandResult = json.toJavaObject(CommandResult.StringExtra.class);
                    break;
                case FETCH_PROGRAMS:
                    commandResult = json.toJavaObject(CommandResult.ProgramExtra.class);
                    break;
                default:
                    commandResult = json.toJavaObject(CommandResult.NoneExtra.class);
            }
        } else {
            commandResult = json.toJavaObject(CommandResult.NoneExtra.class);
        }
        return commandResult;
    }
}
