/*
 * Copyright 2002-2004 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.transaction.interceptor;

import java.lang.reflect.Method;
import java.util.Collection;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.metadata.Attributes;
import org.springframework.util.Assert;

/**
 * Implementation of TransactionAttributeSource that uses
 * attributes from an Attributes implementation.
 * @author Rod Johnson
 * @author Juergen Hoeller
 * @see org.springframework.metadata.Attributes
 * @see org.springframework.transaction.interceptor.AbstractFallbackTransactionAttributeSource
 */
public class AttributesTransactionAttributeSource extends AbstractFallbackTransactionAttributeSource
		implements InitializingBean {
	
	/**
	 * Underlying Attributes implementation that we're using.
	 */
	private Attributes attributes;

	/**
	 * Create a new AttributesTransactionAttributeSource.
	 * @see #setAttributes
	 */
	public AttributesTransactionAttributeSource() {
	}

	/**
	 * Create a new AttributesTransactionAttributeSource.
	 * @param attributes the Attributes implementation to use
	 */
	public AttributesTransactionAttributeSource(Attributes attributes) {
		this.attributes = attributes;
	}

	/**
	 * Set the Attributes implementation to use.
	 */
	public void setAttributes(Attributes attributes) {
		this.attributes = attributes;
	}

	public void afterPropertiesSet() {
		if (this.attributes == null) {
			throw new IllegalArgumentException("'attributes' is required");
		}
	}

	protected Collection findAllAttributes(Class clazz) {
		Assert.notNull(this.attributes, "'attributes' is required");
		return this.attributes.getAttributes(clazz);
	}

	protected Collection findAllAttributes(Method method) {
		Assert.notNull(this.attributes, "'attributes' is required");
		return this.attributes.getAttributes(method);
	}

}
