package ir.afraapps.basic.view;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.PorterDuff;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.support.annotation.NonNull;
import android.support.v4.view.ViewCompat;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import android.widget.LinearLayout;

/**
 * @author ali jabbari on 11/9/2016.
 */

public class DividerRecyclerViewItem extends RecyclerView.ItemDecoration {

  public static final int HORIZONTAL = LinearLayout.HORIZONTAL;
  public static final int VERTICAL   = LinearLayout.VERTICAL;

  private static final int[] ATTRS = new int[]{android.R.attr.listDivider};

  private Drawable mDivider;
  private int marginLeft, marginTop, marginRight, marginBottom = 0;
  private int mColor        = 0;
  private int width, height = 0;

  /**
   * Current orientation. Either {@link #HORIZONTAL} or {@link #VERTICAL}.
   */
  private int mOrientation;

  private final Rect mBounds = new Rect();

  /**
   * Creates a divider {@link RecyclerView.ItemDecoration} that can be used with a
   * {@link LinearLayoutManager}.
   *
   * @param context     Current context, it will be used to access resources.
   * @param orientation Divider orientation. Should be {@link #HORIZONTAL} or {@link #VERTICAL}.
   */
  public DividerRecyclerViewItem(Context context, int orientation) {
    final TypedArray a = context.obtainStyledAttributes(ATTRS);
    mDivider = a.getDrawable(0);
    a.recycle();
    setOrientation(orientation);
  }


  /**
   * Sets the orientation for this divider. This should be called if
   * {@link RecyclerView.LayoutManager} changes orientation.
   *
   * @param orientation {@link #HORIZONTAL} or {@link #VERTICAL}
   */
  public void setOrientation(int orientation) {
    if (orientation != HORIZONTAL && orientation != VERTICAL) {
      throw new IllegalArgumentException(
        "Invalid orientation. It should be either HORIZONTAL or VERTICAL");
    }
    mOrientation = orientation;
  }

  /**
   * Sets the {@link Drawable} for this divider.
   *
   * @param drawable Drawable that should be used as a divider.
   */
  public void setDrawable(@NonNull Drawable drawable) {
    mDivider = drawable;
  }


  /**
   * @return the height of divider in vertical type
   */
  public int getHeight() {
    return height;
  }


  /**
   * @param height the height of divider in vertical type
   */
  public void setHeight(int height) {
    this.height = height;
  }


  /**
   * @return the width of divider in horizontal type
   */
  public int getWidth() {
    return width;
  }


  /**
   * @param width the width of divider in horizontal type
   */
  public void setWidth(int width) {
    this.width = width;
  }

  @Override
  public void onDraw(Canvas c, RecyclerView parent, RecyclerView.State state) {
    if (parent.getLayoutManager() == null) {
      return;
    }
    if (mOrientation == VERTICAL) {
      drawVertical(c, parent);
    } else {
      drawHorizontal(c, parent);
    }
  }

  @SuppressLint("NewApi")
  private void drawVertical(Canvas canvas, RecyclerView parent) {
    canvas.save();
    final int left;
    final int right;
    if (parent.getClipToPadding()) {
      left = parent.getPaddingLeft() + marginLeft;
      right = parent.getWidth() - parent.getPaddingRight() - marginRight;
      canvas.clipRect(left, parent.getPaddingTop(), right,
        parent.getHeight() - parent.getPaddingBottom());
    } else {
      left = marginLeft;
      right = parent.getWidth() - marginRight;
    }

    if (mColor != 0) {
      mDivider.setColorFilter(mColor, PorterDuff.Mode.SRC_ATOP);
    }

    final int childCount = parent.getChildCount() - 1;
    for (int i = 0; i < childCount; i++) {
      final View child = parent.getChildAt(i);
      parent.getDecoratedBoundsWithMargins(child, mBounds);
      final int bottom = mBounds.bottom + Math.round(ViewCompat.getTranslationY(child)) - marginBottom;
      final int top    = bottom - (height == 0 ? mDivider.getIntrinsicHeight() : height);
      mDivider.setBounds(left, top, right, bottom);

      mDivider.draw(canvas);
    }
    canvas.restore();
  }

  @SuppressLint("NewApi")
  private void drawHorizontal(Canvas canvas, RecyclerView parent) {
    canvas.save();
    final int top;
    final int bottom;
    if (parent.getClipToPadding()) {
      top = parent.getPaddingTop() + marginTop;
      bottom = parent.getHeight() - parent.getPaddingBottom() - marginBottom;
      canvas.clipRect(parent.getPaddingLeft(), top,
        parent.getWidth() - parent.getPaddingRight(), bottom);
    } else {
      top = marginTop;
      bottom = parent.getHeight() - marginBottom;
    }

    if (mColor != 0) {
      mDivider.setColorFilter(mColor, PorterDuff.Mode.SRC_ATOP);
    }

    final int childCount = parent.getChildCount() - 1;
    for (int i = 0; i < childCount; i++) {
      final View child = parent.getChildAt(i);
      parent.getLayoutManager().getDecoratedBoundsWithMargins(child, mBounds);
      final int right = mBounds.right + Math.round(ViewCompat.getTranslationX(child)) - marginRight;
      final int left  = right - (width == 0 ? mDivider.getIntrinsicWidth() : width);
      mDivider.setBounds(left, top, right, bottom);

      mDivider.draw(canvas);
    }
    canvas.restore();
  }

  @Override
  public void getItemOffsets(Rect outRect, View view, RecyclerView parent,
                             RecyclerView.State state) {

    if (mOrientation == VERTICAL) {
      outRect.set(0, 0, 0, mDivider.getIntrinsicHeight() + marginBottom + marginTop);
    } else {
      outRect.set(0, 0, mDivider.getIntrinsicWidth() + marginLeft + marginRight, 0);
    }
  }


  /**
   * Sets the color for this divider.
   *
   * @param color int that used as a divider color.
   */
  public void setDividerColor(int color) {
    mColor = color;
  }


  /**
   * Sets the {@link android.view.ViewGroup.MarginLayoutParams} for this divider.
   *
   * @param left   int that used as a divider left margin.
   * @param top    int that used as a divider top margin.
   * @param right  int that used as a divider right margin.
   * @param bottom int that used as a divider bottom margin.
   */
  public void setMargins(int left, int top, int right, int bottom) {
    this.marginLeft = left;
    this.marginTop = top;
    this.marginRight = right;
    this.marginBottom = bottom;

  }

  /**
   * Sets the {@link android.view.ViewGroup.MarginLayoutParams} for this divider.
   *
   * @param margin int that used as a divider left, top, right and bottom margin.
   */
  public void setMargins(int margin) {
    this.marginLeft = margin;
    this.marginTop = margin;
    this.marginRight = margin;
    this.marginBottom = margin;

  }

  public void setMarginLeft(int marginLeft) {
    this.marginLeft = marginLeft;
  }

  public void setMarginTop(int marginTop) {
    this.marginTop = marginTop;
  }

  public void setMarginRight(int marginRight) {
    this.marginRight = marginRight;
  }

  public void setMarginBottom(int marginBottom) {
    this.marginBottom = marginBottom;
  }
}