package ir.afraapps.basic.module.calendar;

import java.util.Calendar;


public class PersianDate extends AbstractDate {

  private static final String[] persianMonthName = {"", "فروردین",
    "اردیبهشت", "خرداد", "تیر", "مرداد", "شهریور", "مهر", "آبان",
    "آذر", "دی", "بهمن", "اسفند"};

  private static final String[] persianDayName = {"", "یکشنبه",
    "دوشنبه", "سه‌شنبه", "چهارشنبه",
    "پنج‌شنبه", "جمعه", "شنبه"};

  private static final int[] daysOfMonth = {0, 31, 31, 31, 31, 31, 31, 30, 30, 30, 30, 30, 29};


  @Override
  public String[] getMonthsList() {
    return persianMonthName;
  }

  private int year;
  private int month;
  private int day;


  public PersianDate(int year, int month, int day) {
    setYear(year);
    // Initialize day, so that we get no exceptions when setting month
    this.day = 1;
    setMonth(month);
    setDayOfMonth(day);
  }


  @Override
  public PersianDate clone() {
    return new PersianDate(getYear(), getMonth(), getDayOfMonth());
  }


  @Override
  public int getDayOfMonth() {
    return day;
  }


  public int getDaysInMonth(int year, int month) {
    setYear(year);
    setMonth(month);
    setDayOfMonth(1);
    if (isLeapYear() && month == 12) {
      return 30;
    } else {
      return daysOfMonth[month];
    }
  }


  public int getDaysInMonth() {
    if (month == 12 && isLeapYear()) {
      return 30;
    } else {
      return daysOfMonth[month];
    }
  }


  public int getDaysInYear() {
    if (isLeapYear()) {
      return 366;
    } else {
      return 365;
    }
  }


  @Override
  public int getMonth() {
    return month;
  }


  @Override
  public String getMonthName() {
    return getMonthsList()[month];
  }


  @Override
  public int getWeekOfYear() {
    throw new RuntimeException("not implemented yet!");
  }


  @Override
  public int getYear() {
    return year;
  }


  @Override
  public void rollDay(int amount, boolean up) {
    throw new RuntimeException("not implemented yet!");
  }


  @Override
  public void rollMonth(int amount, boolean up) {
    throw new RuntimeException("not implemented yet!");
  }


  @Override
  public void rollYear(int amount, boolean up) {
    throw new RuntimeException("not implemented yet!");
  }


  @Override
  public void setDayOfMonth(int day) {
    if (day < 1)
      throw new DayOutOfRangeException("day " + day + " is out of range!");

    if (month <= 6 && day > 31)
      throw new DayOutOfRangeException("day " + day + " is out of range!");

    if (month > 6 && month <= 12 && day > 30)
      throw new DayOutOfRangeException("day " + day + " is out of range!");

    if (isLeapYear() && month == 12 && day > 30)
      throw new DayOutOfRangeException("day " + day + " is out of range!");

    if ((!isLeapYear()) && month == 12 && day > 29)
      throw new DayOutOfRangeException("day " + day + " is out of range!");

    this.day = day;
  }


  @Override
  public void setMonth(int month) {
    if (month < 1 || month > 12)
      throw new MonthOutOfRangeException("month " + month
        + " is out of range!");

    // Set the day again, so that exceptions are thrown if the
    // day is out of range
    setDayOfMonth(day);

    this.month = month;
  }


  @Override
  public void setYear(int year) {
    if (year == 0)
      throw new YearOutOfRangeException("Year 0 is invalid!");

    this.year = year;
  }


  @Override
  public String getEvent() {
    throw new RuntimeException("not implemented yet!");
  }


  @Override
  public int getDayOfWeek() {
    PersianDate persian = this.clone();
    return DateConverter.persianToCalendar(persian).get(Calendar.DAY_OF_WEEK);
  }


  public String getDayName(int day) {
    return persianDayName[day];
  }


  public String getDayName() {
    return persianDayName[getDayOfWeek()];
  }


  @Override
  public int getDayOfYear() {
    throw new RuntimeException("not implemented yet!");
  }


  @Override
  public int getWeekOfMonth() {
    PersianDate persian = clone();
    persian.setDayOfMonth(1);
    int dayOfWeek = persian.getDayOfWeek() % 7;
    return 1 + ((day + dayOfWeek) / 7);
  }


  public int getWeeksOfMonth() {
    PersianDate persian = clone();
    persian.setDayOfMonth(1);
    int dayOfWeek = persian.getDayOfWeek() % 7;
    return 1 + ((getDaysInMonth() + dayOfWeek) / 7);
  }


  @Override
  public boolean isLeapYear() {
    int y;
    if (year > 0)
      y = year - 474;
    else
      y = 473;
    return (((((y % 2820) + 474) + 38) * 682) % 2816) < 682;
  }


  public static boolean isLipYear(int year) {
    int y;
    if (year > 0)
      y = year - 474;
    else
      y = 473;
    return (((((y % 2820) + 474) + 38) * 682) % 2816) < 682;
  }


  public boolean isLeapYear(PersianDate persian) {
    int y;
    if (persian.getYear() > 0)
      y = persian.getYear() - 474;
    else
      y = 473;
    return (((((y % 2820) + 474) + 38) * 682) % 2816) < 682;
  }


  public boolean equals(PersianDate persianDate) {
    if (this.getDayOfMonth() == persianDate.getDayOfMonth()
      && this.getMonth() == persianDate.getMonth()
      && this.getYear() == persianDate.getYear())
      return true;
    return false;
  }
}
