/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-bridge-common-js/base_bridge_event */
var utils = require('vertx-js/util/utils');
var Future = require('vertx-js/future');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JBaseBridgeEvent = Java.type('io.vertx.ext.bridge.BaseBridgeEvent');

/**
 Represents an event that occurs on the event bus bridge.
 <p>
 Please consult the documentation for a full explanation.

 @class
*/
var BaseBridgeEvent = function(j_val) {

  var j_baseBridgeEvent = j_val;
  var that = this;
  Future.call(this, j_val, undefined);

  var __super_setHandler = this.setHandler;
  var __super_complete = this.complete;
  var __super_tryComplete = this.tryComplete;
  var __super_result = this.result;
  var __super_compose = this.compose;
  var __super_compose = this.compose;
  var __super_map = this.map;
  var __super_completer = this.completer;
  var __super_recover = this.recover;
  var __super_otherwise = this.otherwise;
  var __super_otherwise = this.otherwise;
  var __super_otherwiseEmpty = this.otherwiseEmpty;
  var __super_type = this.type;
  var __super_getRawMessage = this.getRawMessage;
  var __super_setRawMessage = this.setRawMessage;
  /**
   Set a handler for the result.
   <p>
   If the future has already been completed it will be called immediately. Otherwise it will be called when the
   future is completed.

   @public
   @param handler {function} the Handler that will be called with the result 
   @return {Future} a reference to this, so it can be used fluently
   */
  this.setHandler =  function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_baseBridgeEvent["setHandler(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __args[0](ar.result(), null);
        } else {
          __args[0](null, ar.cause());
        }
      }) ;
      return that;
    } else if (typeof __super_setHandler != 'undefined') {
      return __super_setHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the result. Any handler will be called, if there is one, and the future will be marked as completed.

   @public
   @param result {boolean} the result 
   */
  this.complete =  function(result) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      j_baseBridgeEvent["complete(java.lang.Boolean)"](__args[0]);
    } else if (typeof __super_complete != 'undefined') {
      return __super_complete.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Set the failure. Any handler will be called, if there is one, and the future will be marked as completed.

   @public
   @param result {boolean} the result 
   @return {boolean} false when the future is already completed
   */
  this.tryComplete =  function(result) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      return j_baseBridgeEvent["tryComplete(java.lang.Boolean)"](__args[0]) ;
    } else if (typeof __super_tryComplete != 'undefined') {
      return __super_tryComplete.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   The result of the operation. This will be null if the operation failed.

   @public

   @return {boolean} the result or null if the operation failed.
   */
  this.result =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_baseBridgeEvent["result()"]() ;
    } else if (typeof __super_result != 'undefined') {
      return __super_result.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Compose this future with a <code>mapper</code> function.<p>

   When this future (the one on which <code>compose</code> is called) succeeds, the <code>mapper</code> will be called with
   the completed value and this mapper returns another future object. This returned future completion will complete
   the future returned by this method call.<p>

   If the <code>mapper</code> throws an exception, the returned future will be failed with this exception.<p>

   When this future fails, the failure will be propagated to the returned future and the <code>mapper</code>
   will not be called.

   @public
   @param mapper {function} the mapper function 
   @return {Future} the composed future
   */
  this.compose =  function() {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'function' && typeof __args[1] === 'object' && __args[1]._jdel) {
      return utils.convReturnVertxGen(Future, j_baseBridgeEvent["compose(io.vertx.core.Handler,io.vertx.core.Future)"](function(jVal) {
        __args[0](jVal);
      }, __args[1]._jdel), undefined) ;
    } else if (__args.length === 1 && typeof __args[0] === 'function') {
      return utils.convReturnVertxGen(Future, j_baseBridgeEvent["compose(java.util.function.Function)"](function(jVal) {
        var jRet = __args[0](jVal);
        return jRet._jdel;
      }), undefined) ;
    } else if (typeof __super_compose != 'undefined') {
      return __super_compose.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Apply a <code>mapper</code> function on this future.<p>

   When this future succeeds, the <code>mapper</code> will be called with the completed value and this mapper
   returns a value. This value will complete the future returned by this method call.<p>

   If the <code>mapper</code> throws an exception, the returned future will be failed with this exception.<p>

   When this future fails, the failure will be propagated to the returned future and the <code>mapper</code>
   will not be called.

   @public
   @param mapper {function} the mapper function 
   @return {Future} the mapped future
   */
  this.map =  function(mapper) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      return utils.convReturnVertxGen(Future, j_baseBridgeEvent["map(java.util.function.Function)"](function(jVal) {
        var jRet = __args[0](jVal);
        return utils.convParamTypeUnknown(jRet);
      }), undefined) ;
    } else if (typeof __super_map != 'undefined') {
      return __super_map.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {function} an handler completing this future
   */
  this.completer =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedcompleter == null) {
        that.cachedcompleter = utils.convReturnHandlerAsyncResult(j_baseBridgeEvent["completer()"](), function(result) { return result; });
      }
      return that.cachedcompleter;
    } else if (typeof __super_completer != 'undefined') {
      return __super_completer.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Handles a failure of this Future by returning the result of another Future.
   If the mapper fails, then the returned future will be failed with this failure.

   @public
   @param mapper {function} A function which takes the exception of a failure and returns a new future. 
   @return {Future} A recovered future
   */
  this.recover =  function(mapper) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      return utils.convReturnVertxGen(Future, j_baseBridgeEvent["recover(java.util.function.Function)"](function(jVal) {
        var jRet = __args[0](utils.convReturnThrowable(jVal));
        return jRet._jdel;
      }), undefined) ;
    } else if (typeof __super_recover != 'undefined') {
      return __super_recover.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Map the failure of a future to a specific <code>value</code>.<p>

   When this future fails, this <code>value</code> will complete the future returned by this method call.<p>

   When this future succeeds, the result will be propagated to the returned future.

   @public
   @param value {boolean} the value that eventually completes the mapped future 
   @return {Future} the mapped future
   */
  this.otherwise =  function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      return utils.convReturnVertxGen(Future, j_baseBridgeEvent["otherwise(java.util.function.Function)"](function(jVal) {
        var jRet = __args[0](utils.convReturnThrowable(jVal));
        return jRet;
      }), undefined) ;
    } else if (__args.length === 1 && typeof __args[0] ==='boolean') {
      return utils.convReturnVertxGen(Future, j_baseBridgeEvent["otherwise(java.lang.Boolean)"](__args[0]), undefined) ;
    } else if (typeof __super_otherwise != 'undefined') {
      return __super_otherwise.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Map the failure of a future to <code>null</code>.<p>

   This is a convenience for <code>future.otherwise((T) null)</code>.<p>

   When this future fails, the <code>null</code> value will complete the future returned by this method call.<p>

   When this future succeeds, the result will be propagated to the returned future.

   @public

   @return {Future} the mapped future
   */
  this.otherwiseEmpty =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Future, j_baseBridgeEvent["otherwiseEmpty()"](), undefined) ;
    } else if (typeof __super_otherwiseEmpty != 'undefined') {
      return __super_otherwiseEmpty.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {Object} the type of the event
   */
  this.type =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedtype == null) {
        that.cachedtype = utils.convReturnEnum(j_baseBridgeEvent["type()"]());
      }
      return that.cachedtype;
    } else if (typeof __super_type != 'undefined') {
      return __super_type.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the raw JSON message for the event. This will be null for SOCKET_CREATED or SOCKET_CLOSED events as there is
   no message involved. If the returned message is modified, {@link BaseBridgeEvent#setRawMessage} should be called with the
   new message.

   @public

   @return {Object} the raw JSON message for the event
   */
  this.getRawMessage =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnJson(j_baseBridgeEvent["getRawMessage()"]()) ;
    } else if (typeof __super_getRawMessage != 'undefined') {
      return __super_getRawMessage.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get the raw JSON message for the event. This will be null for SOCKET_CREATED or SOCKET_CLOSED events as there is
   no message involved.

   @public
   @param message {Object} the raw message 
   @return {BaseBridgeEvent} this reference, so it can be used fluently
   */
  this.setRawMessage =  function(message) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'object' && __args[0] != null)) {
      j_baseBridgeEvent["setRawMessage(io.vertx.core.json.JsonObject)"](utils.convParamJsonObject(__args[0])) ;
      return that;
    } else if (typeof __super_setRawMessage != 'undefined') {
      return __super_setRawMessage.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_baseBridgeEvent;
};

BaseBridgeEvent._jclass = utils.getJavaClass("io.vertx.ext.bridge.BaseBridgeEvent");
BaseBridgeEvent._jtype = {accept: function(obj) {
    return BaseBridgeEvent._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(BaseBridgeEvent.prototype, {});
    BaseBridgeEvent.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
BaseBridgeEvent._create = function(jdel) {var obj = Object.create(BaseBridgeEvent.prototype, {});
  BaseBridgeEvent.apply(obj, arguments);
  return obj;
}
module.exports = BaseBridgeEvent;