/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.rxjava.core.http;

import java.util.Map;
import rx.Observable;
import rx.Single;
import io.vertx.rxjava.core.MultiMap;
import io.vertx.core.http.RequestOptions;
import io.vertx.core.http.WebsocketVersion;
import java.util.function.Function;
import io.vertx.rxjava.core.Future;
import io.vertx.rxjava.core.metrics.Measured;
import io.vertx.core.http.HttpMethod;
import io.vertx.rxjava.core.streams.ReadStream;
import io.vertx.core.Handler;

/**
 * An asynchronous HTTP client.
 * <p>
 * It allows you to make requests to HTTP servers, and a single client can make requests to any server.
 * <p>
 * It also allows you to open WebSockets to servers.
 * <p>
 * The client can also pool HTTP connections.
 * <p>
 * For pooling to occur, keep-alive must be true on the {@link io.vertx.core.http.HttpClientOptions} (default is true).
 * In this case connections will be pooled and re-used if there are pending HTTP requests waiting to get a connection,
 * otherwise they will be closed.
 * <p>
 * This gives the benefits of keep alive when the client is loaded but means we don't keep connections hanging around
 * unnecessarily when there would be no benefits anyway.
 * <p>
 * The client also supports pipe-lining of requests. Pipe-lining means another request is sent on the same connection
 * before the response from the preceding one has returned. Pipe-lining is not appropriate for all requests.
 * <p>
 * To enable pipe-lining, it must be enabled on the {@link io.vertx.core.http.HttpClientOptions} (default is false).
 * <p>
 * When pipe-lining is enabled the connection will be automatically closed when all in-flight responses have returned
 * and there are no outstanding pending requests to write.
 * <p>
 * The client is designed to be reused between requests.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpClient original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.rxjava.RxGen(io.vertx.core.http.HttpClient.class)
public class HttpClient implements Measured {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpClient that = (HttpClient) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final io.vertx.lang.rxjava.TypeArg<HttpClient> __TYPE_ARG = new io.vertx.lang.rxjava.TypeArg<>(
    obj -> new HttpClient((io.vertx.core.http.HttpClient) obj),
    HttpClient::getDelegate
  );

  private final io.vertx.core.http.HttpClient delegate;
  
  public HttpClient(io.vertx.core.http.HttpClient delegate) {
    this.delegate = delegate;
  }

  public io.vertx.core.http.HttpClient getDelegate() {
    return delegate;
  }

  /**
   * Whether the metrics are enabled for this measured object
   * @return true if the metrics are enabled
   */
  public boolean isMetricsEnabled() { 
    boolean ret = delegate.isMetricsEnabled();
    return ret;
  }

  /**
   * Create an HTTP request to send to the server with the specified options.
   * @param method the HTTP method
   * @param options the request options
   * @return an HTTP client request object
   */
  public HttpClientRequest request(HttpMethod method, RequestOptions options) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.request(method, options));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the specified host and port.
   * @param method the HTTP method
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest request(HttpMethod method, int port, String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.request(method, port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the specified host and default port.
   * @param method the HTTP method
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest request(HttpMethod method, String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.request(method, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server with the specified options, specifying a response handler to receive
   * @param method the HTTP method
   * @param options the request options
   * @param responseHandler 
   * @return an HTTP client request object
   */
  public HttpClientRequest request(HttpMethod method, RequestOptions options, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.request(method, options, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param method the HTTP method
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest request(HttpMethod method, int port, String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.request(method, port, host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param method the HTTP method
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest request(HttpMethod method, String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.request(method, host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the default host and port.
   * @param method the HTTP method
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest request(HttpMethod method, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.request(method, requestURI));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param method the HTTP method
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest request(HttpMethod method, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.request(method, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server using an absolute URI
   * @param method the HTTP method
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public HttpClientRequest requestAbs(HttpMethod method, String absoluteURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.requestAbs(method, absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param method the HTTP method
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest requestAbs(HttpMethod method, String absoluteURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.requestAbs(method, absoluteURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public HttpClientRequest get(RequestOptions options) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.get(options));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest get(int port, String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.get(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest get(String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.get(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest get(RequestOptions options, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.get(options, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest get(int port, String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.get(port, host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest get(String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.get(host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest get(String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.get(requestURI));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest get(String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.get(requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public HttpClientRequest getAbs(String absoluteURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.getAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP GET request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest getAbs(String absoluteURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.getAbs(absoluteURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Sends an HTTP GET request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient getNow(RequestOptions options, Handler<HttpClientResponse> responseHandler) { 
    delegate.getNow(options, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Sends an HTTP GET request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient getNow(int port, String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    delegate.getNow(port, host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Sends an HTTP GET request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient getNow(String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    delegate.getNow(host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Sends an HTTP GET request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient getNow(String requestURI, Handler<HttpClientResponse> responseHandler) { 
    delegate.getNow(requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Create an HTTP POST request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public HttpClientRequest post(RequestOptions options) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.post(options));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest post(int port, String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.post(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest post(String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.post(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest post(RequestOptions options, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.post(options, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest post(int port, String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.post(port, host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest post(String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.post(host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest post(String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.post(requestURI));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest post(String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.post(requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public HttpClientRequest postAbs(String absoluteURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.postAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP POST request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest postAbs(String absoluteURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.postAbs(absoluteURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public HttpClientRequest head(RequestOptions options) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.head(options));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest head(int port, String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.head(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest head(String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.head(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest head(RequestOptions options, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.head(options, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest head(int port, String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.head(port, host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest head(String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.head(host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest head(String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.head(requestURI));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest head(String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.head(requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public HttpClientRequest headAbs(String absoluteURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.headAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP HEAD request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest headAbs(String absoluteURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.headAbs(absoluteURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Sends an HTTP HEAD request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient headNow(RequestOptions options, Handler<HttpClientResponse> responseHandler) { 
    delegate.headNow(options, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Sends an HTTP HEAD request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient headNow(int port, String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    delegate.headNow(port, host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Sends an HTTP HEAD request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient headNow(String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    delegate.headNow(host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Sends an HTTP HEAD request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient headNow(String requestURI, Handler<HttpClientResponse> responseHandler) { 
    delegate.headNow(requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public HttpClientRequest options(RequestOptions options) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.options(options));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest options(int port, String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.options(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest options(String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.options(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest options(RequestOptions options, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.options(options, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest options(int port, String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.options(port, host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest options(String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.options(host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest options(String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.options(requestURI));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest options(String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.options(requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public HttpClientRequest optionsAbs(String absoluteURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.optionsAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP OPTIONS request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest optionsAbs(String absoluteURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.optionsAbs(absoluteURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Sends an HTTP OPTIONS request to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient optionsNow(RequestOptions options, Handler<HttpClientResponse> responseHandler) { 
    delegate.optionsNow(options, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Sends an HTTP OPTIONS request to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient optionsNow(int port, String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    delegate.optionsNow(port, host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Sends an HTTP OPTIONS request to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient optionsNow(String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    delegate.optionsNow(host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Sends an HTTP OPTIONS request  to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient optionsNow(String requestURI, Handler<HttpClientResponse> responseHandler) { 
    delegate.optionsNow(requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Create an HTTP PUT request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public HttpClientRequest put(RequestOptions options) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.put(options));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest put(int port, String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.put(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest put(String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.put(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest put(RequestOptions options, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.put(options, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest put(int port, String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.put(port, host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest put(String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.put(host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest put(String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.put(requestURI));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest put(String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.put(requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public HttpClientRequest putAbs(String absoluteURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.putAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP PUT request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest putAbs(String absoluteURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.putAbs(absoluteURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server with the specified options.
   * @param options the request options
   * @return an HTTP client request object
   */
  public HttpClientRequest delete(RequestOptions options) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.delete(options));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the specified host and port.
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest delete(int port, String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.delete(port, host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the specified host and default port.
   * @param host the host
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest delete(String host, String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.delete(host, requestURI));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server with the specified options, specifying a response handler to receive
   * the response
   * @param options the request options
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest delete(RequestOptions options, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.delete(options, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the specified host and port, specifying a response handler to receive
   * the response
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest delete(int port, String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.delete(port, host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the specified host and default port, specifying a response handler to receive
   * the response
   * @param host the host
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest delete(String host, String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.delete(host, requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the default host and port.
   * @param requestURI the relative URI
   * @return an HTTP client request object
   */
  public HttpClientRequest delete(String requestURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.delete(requestURI));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server at the default host and port, specifying a response handler to receive
   * the response
   * @param requestURI the relative URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest delete(String requestURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.delete(requestURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server using an absolute URI
   * @param absoluteURI the absolute URI
   * @return an HTTP client request object
   */
  public HttpClientRequest deleteAbs(String absoluteURI) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.deleteAbs(absoluteURI));
    return ret;
  }

  /**
   * Create an HTTP DELETE request to send to the server using an absolute URI, specifying a response handler to receive
   * the response
   * @param absoluteURI the absolute URI
   * @param responseHandler the response handler
   * @return an HTTP client request object
   */
  public HttpClientRequest deleteAbs(String absoluteURI, Handler<HttpClientResponse> responseHandler) { 
    HttpClientRequest ret = HttpClientRequest.newInstance(delegate.deleteAbs(absoluteURI, new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        responseHandler.handle(HttpClientResponse.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Connect a WebSocket with the specified options
   * @param options the request options
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(RequestOptions options, Handler<WebSocket> wsConnect) { 
    delegate.websocket(options, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(int port, String host, String requestURI, Handler<WebSocket> wsConnect) { 
    delegate.websocket(port, host, requestURI, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket with the specified options
   * @param options the request options
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(RequestOptions options, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(options, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(int port, String host, String requestURI, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(port, host, requestURI, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket to the host and relative request URI and default port
   * @param host the host
   * @param requestURI the relative URI
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String host, String requestURI, Handler<WebSocket> wsConnect) { 
    delegate.websocket(host, requestURI, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket to the host and relative request URI and default port
   * @param host the host
   * @param requestURI the relative URI
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String host, String requestURI, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(host, requestURI, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket with the specified options, and with the specified headers
   * @param options the request options
   * @param headers the headers
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(RequestOptions options, MultiMap headers, Handler<WebSocket> wsConnect) { 
    delegate.websocket(options, headers.getDelegate(), new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI, and with the specified headers
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(int port, String host, String requestURI, MultiMap headers, Handler<WebSocket> wsConnect) { 
    delegate.websocket(port, host, requestURI, headers.getDelegate(), new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket with the specified options, and with the specified headers
   * @param options the request options
   * @param headers the headers
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(RequestOptions options, MultiMap headers, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(options, headers.getDelegate(), new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI, and with the specified headers
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(int port, String host, String requestURI, MultiMap headers, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(port, host, requestURI, headers.getDelegate(), new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket to the specified host,relative request UR, and default port and with the specified headers
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String host, String requestURI, MultiMap headers, Handler<WebSocket> wsConnect) { 
    delegate.websocket(host, requestURI, headers.getDelegate(), new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket to the specified host,relative request UR, and default port and with the specified headers
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String host, String requestURI, MultiMap headers, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(host, requestURI, headers.getDelegate(), new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket with the specified optionsI, with the specified headers and using
   * the specified version of WebSockets
   * @param options the request options
   * @param headers the headers
   * @param version the websocket version
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(RequestOptions options, MultiMap headers, WebsocketVersion version, Handler<WebSocket> wsConnect) { 
    delegate.websocket(options, headers.getDelegate(), version, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI, with the specified headers and using
   * the specified version of WebSockets
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(int port, String host, String requestURI, MultiMap headers, WebsocketVersion version, Handler<WebSocket> wsConnect) { 
    delegate.websocket(port, host, requestURI, headers.getDelegate(), version, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket with the specified options, with the specified headers and using
   * the specified version of WebSockets
   * @param options the request options
   * @param headers the headers
   * @param version the websocket version
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(RequestOptions options, MultiMap headers, WebsocketVersion version, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(options, headers.getDelegate(), version, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI, with the specified headers and using
   * the specified version of WebSockets
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(int port, String host, String requestURI, MultiMap headers, WebsocketVersion version, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(port, host, requestURI, headers.getDelegate(), version, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket to the specified host, relative request URI and default port with the specified headers and using
   * the specified version of WebSockets
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String host, String requestURI, MultiMap headers, WebsocketVersion version, Handler<WebSocket> wsConnect) { 
    delegate.websocket(host, requestURI, headers.getDelegate(), version, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket to the specified host, relative request URI and default port with the specified headers and using
   * the specified version of WebSockets
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String host, String requestURI, MultiMap headers, WebsocketVersion version, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(host, requestURI, headers.getDelegate(), version, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket with the specified options, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols
   * @param options the request options
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(RequestOptions options, MultiMap headers, WebsocketVersion version, String subProtocols, Handler<WebSocket> wsConnect) { 
    delegate.websocket(options, headers.getDelegate(), version, subProtocols, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(int port, String host, String requestURI, MultiMap headers, WebsocketVersion version, String subProtocols, Handler<WebSocket> wsConnect) { 
    delegate.websocket(port, host, requestURI, headers.getDelegate(), version, subProtocols, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket with the specified absolute url, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols.
   * @param url the absolute url
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocketAbs(String url, MultiMap headers, WebsocketVersion version, String subProtocols, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocketAbs(url, headers.getDelegate(), version, subProtocols, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket with the specified options, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols
   * @param options the request options
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(RequestOptions options, MultiMap headers, WebsocketVersion version, String subProtocols, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(options, headers.getDelegate(), version, subProtocols, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(int port, String host, String requestURI, MultiMap headers, WebsocketVersion version, String subProtocols, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(port, host, requestURI, headers.getDelegate(), version, subProtocols, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket to the specified host, relative request URI and default port, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String host, String requestURI, MultiMap headers, WebsocketVersion version, String subProtocols, Handler<WebSocket> wsConnect) { 
    delegate.websocket(host, requestURI, headers.getDelegate(), version, subProtocols, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket to the specified host, relative request URI and default port, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String host, String requestURI, MultiMap headers, WebsocketVersion version, String subProtocols, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(host, requestURI, headers.getDelegate(), version, subProtocols, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port
   * @param requestURI the relative URI
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String requestURI, Handler<WebSocket> wsConnect) { 
    delegate.websocket(requestURI, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port
   * @param requestURI the relative URI
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String requestURI, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(requestURI, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port and the specified headers
   * @param requestURI the relative URI
   * @param headers the headers
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String requestURI, MultiMap headers, Handler<WebSocket> wsConnect) { 
    delegate.websocket(requestURI, headers.getDelegate(), new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port and the specified headers
   * @param requestURI the relative URI
   * @param headers the headers
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String requestURI, MultiMap headers, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(requestURI, headers.getDelegate(), new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port, the specified headers and the
   * specified version of WebSockets
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String requestURI, MultiMap headers, WebsocketVersion version, Handler<WebSocket> wsConnect) { 
    delegate.websocket(requestURI, headers.getDelegate(), version, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port, the specified headers and the
   * specified version of WebSockets
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String requestURI, MultiMap headers, WebsocketVersion version, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(requestURI, headers.getDelegate(), version, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port, the specified headers, the
   * specified version of WebSockets and the specified sub protocols
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols
   * @param wsConnect handler that will be called with the websocket when connected
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String requestURI, MultiMap headers, WebsocketVersion version, String subProtocols, Handler<WebSocket> wsConnect) { 
    delegate.websocket(requestURI, headers.getDelegate(), version, subProtocols, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port, the specified headers, the
   * specified version of WebSockets and the specified sub protocols
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols
   * @param wsConnect handler that will be called with the websocket when connected
   * @param failureHandler handler that will be called if websocket connection fails
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient websocket(String requestURI, MultiMap headers, WebsocketVersion version, String subProtocols, Handler<WebSocket> wsConnect, Handler<Throwable> failureHandler) { 
    delegate.websocket(requestURI, headers.getDelegate(), version, subProtocols, new Handler<io.vertx.core.http.WebSocket>() {
      public void handle(io.vertx.core.http.WebSocket event) {
        wsConnect.handle(WebSocket.newInstance(event));
      }
    }, failureHandler);
    return this;
  }

  /**
   * Create a WebSocket stream with the specified options
   * @param options the request options
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(RequestOptions options) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(options), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream to the specified port, host and relative request URI
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(int port, String host, String requestURI) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(port, host, requestURI), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream to the specified host, relative request URI and default port
   * @param host the host
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(String host, String requestURI) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(host, requestURI), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream with the specified options, and with the specified headers
   * @param options the request options
   * @param headers the headers
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(RequestOptions options, MultiMap headers) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(options, headers.getDelegate()), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream to the specified port, host and relative request URI, and with the specified headers
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(int port, String host, String requestURI, MultiMap headers) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(port, host, requestURI, headers.getDelegate()), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream to the specified host, relative request URI and default port and with the specified headers
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(String host, String requestURI, MultiMap headers) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(host, requestURI, headers.getDelegate()), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream with the specified options, with the specified headers and using
   * the specified version of WebSockets
   * @param options the request options
   * @param headers the headers
   * @param version the websocket version
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(RequestOptions options, MultiMap headers, WebsocketVersion version) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(options, headers.getDelegate(), version), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream to the specified port, host and relative request URI, with the specified headers and using
   * the specified version of WebSockets
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(int port, String host, String requestURI, MultiMap headers, WebsocketVersion version) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(port, host, requestURI, headers.getDelegate(), version), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream with the specified options and with the specified headers and using
   * the specified version of WebSockets
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(String host, String requestURI, MultiMap headers, WebsocketVersion version) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(host, requestURI, headers.getDelegate(), version), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream with the specified absolute url, the specified headers, using the specified version of WebSockets,
   * and the specified websocket sub protocols.
   * @param url the absolute url
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStreamAbs(String url, MultiMap headers, WebsocketVersion version, String subProtocols) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStreamAbs(url, headers.getDelegate(), version, subProtocols), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream to the specified port, host and relative request URI, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols
   * @param options the request options
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(RequestOptions options, MultiMap headers, WebsocketVersion version, String subProtocols) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(options, headers.getDelegate(), version, subProtocols), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream to the specified port, host and relative request URI, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(int port, String host, String requestURI, MultiMap headers, WebsocketVersion version, String subProtocols) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(port, host, requestURI, headers.getDelegate(), version, subProtocols), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream to the specified host, relative request URI and default port, with the specified headers, using
   * the specified version of WebSockets, and the specified websocket sub protocols
   * @param host the host
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols to use
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(String host, String requestURI, MultiMap headers, WebsocketVersion version, String subProtocols) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(host, requestURI, headers.getDelegate(), version, subProtocols), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream at the relative request URI using the default host and port and the specified headers
   * @param requestURI the relative URI
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(String requestURI) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(requestURI), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream at the relative request URI using the default host and port and the specified headers
   * @param requestURI the relative URI
   * @param headers the headers
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(String requestURI, MultiMap headers) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(requestURI, headers.getDelegate()), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream at the relative request URI using the default host and port, the specified headers and the
   * specified version of WebSockets
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(String requestURI, MultiMap headers, WebsocketVersion version) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(requestURI, headers.getDelegate(), version), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Create a WebSocket stream at the relative request URI using the default host and port, the specified headers, the
   * specified version of WebSockets and the specified sub protocols
   * @param requestURI the relative URI
   * @param headers the headers
   * @param version the websocket version
   * @param subProtocols the subprotocols
   * @return a reference to this, so the API can be used fluently
   */
  public ReadStream<WebSocket> websocketStream(String requestURI, MultiMap headers, WebsocketVersion version, String subProtocols) { 
    ReadStream<WebSocket> ret = ReadStream.newInstance(delegate.websocketStream(requestURI, headers.getDelegate(), version, subProtocols), io.vertx.rxjava.core.http.WebSocket.__TYPE_ARG);
    return ret;
  }

  /**
   * Set a redirect handler for the http client.
   * <p>
   * The redirect handler is called when a <code>3xx</code> response is received and the request is configured to
   * follow redirects with {@link io.vertx.rxjava.core.http.HttpClientRequest#setFollowRedirects}.
   * <p>
   * The redirect handler is passed the {@link io.vertx.rxjava.core.http.HttpClientResponse}, it can return an {@link io.vertx.rxjava.core.http.HttpClientRequest} or <code>null</code>.
   * <ul>
   *   <li>when null is returned, the original response is processed by the original request response handler</li>
   *   <li>when a new <code>Future<HttpClientRequest></code> is returned, the client will send this new request</li>
   * </ul>
   * The handler must return a <code>Future<HttpClientRequest></code> unsent so the client can further configure it and send it.
   * @param handler the new redirect handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClient redirectHandler(Function<HttpClientResponse,Future<HttpClientRequest>> handler) { 
    delegate.redirectHandler(new java.util.function.Function<io.vertx.core.http.HttpClientResponse,io.vertx.core.Future<io.vertx.core.http.HttpClientRequest>>() {
      public io.vertx.core.Future<io.vertx.core.http.HttpClientRequest> apply(io.vertx.core.http.HttpClientResponse arg) {
        Future<HttpClientRequest> ret = handler.apply(HttpClientResponse.newInstance(arg));
        return ret.getDelegate();
      }
    });
    return this;
  }

  /**
   * Close the client. Closing will close down any pooled connections.
   * Clients should always be closed after use.
   */
  public void close() { 
    delegate.close();
  }


  public static  HttpClient newInstance(io.vertx.core.http.HttpClient arg) {
    return arg != null ? new HttpClient(arg) : null;
  }
}
