/*
 * Copyright 2020-Present The Serverless Workflow Specification Authors
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package io.serverlessworkflow.api.deserializers;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import io.serverlessworkflow.api.interfaces.WorkflowPropertySource;
import io.serverlessworkflow.api.retry.RetryDefinition;
import io.serverlessworkflow.api.utils.Utils;
import io.serverlessworkflow.api.workflow.Retries;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public class RetriesDeserializer extends StdDeserializer<Retries> {

    private static final long serialVersionUID = 510l;
    private static Logger logger = LoggerFactory.getLogger(RetriesDeserializer.class);

    private WorkflowPropertySource context;

    public RetriesDeserializer() {
        this(Retries.class);
    }

    public RetriesDeserializer(Class<?> vc) {
        super(vc);
    }

    public RetriesDeserializer(WorkflowPropertySource context) {
        this(Retries.class);
        this.context = context;
    }

    @Override
    public Retries deserialize(JsonParser jp,
                              DeserializationContext ctxt) throws IOException {

        ObjectMapper mapper = (ObjectMapper) jp.getCodec();
        JsonNode node = jp.getCodec().readTree(jp);

        Retries retries = new Retries();
        List<RetryDefinition> retryDefinitions = new ArrayList<>();

        if (node.isArray()) {
            for (final JsonNode nodeEle : node) {
                retryDefinitions.add(mapper.treeToValue(nodeEle, RetryDefinition.class));
            }
        } else {
            String retriesFileDef = node.asText();
            String retriesFileSrc = Utils.getResourceFileAsString(retriesFileDef);
            JsonNode retriesRefNode;
            ObjectMapper jsonWriter = new ObjectMapper();
            if (retriesFileSrc != null && retriesFileSrc.trim().length() > 0) {
                // if its a yaml def convert to json first
                if (!retriesFileSrc.trim().startsWith("{")) {
                    // convert yaml to json to validate
                    ObjectMapper yamlReader = new ObjectMapper(new YAMLFactory());
                    Object obj = yamlReader.readValue(retriesFileSrc, Object.class);

                    retriesRefNode = jsonWriter.readTree(new JSONObject(jsonWriter.writeValueAsString(obj)).toString());
                } else {
                    retriesRefNode = jsonWriter.readTree(new JSONObject(retriesFileSrc).toString());
                }

                JsonNode refRetries = retriesRefNode.get("retries");
                if (refRetries != null) {
                    for (final JsonNode nodeEle : refRetries) {
                        retryDefinitions.add(mapper.treeToValue(nodeEle, RetryDefinition.class));
                    }
                } else {
                    logger.error("Unable to find retries definitions in reference file: {}", retriesFileSrc);
                }

            } else {
                logger.error("Unable to load retries defs reference file: {}", retriesFileSrc);
            }

        }
        retries.setRetryDefs(retryDefinitions);
        return retries;

    }
}
