package io.intercom.android.sdk.identity;

import android.support.annotation.NonNull;
import android.text.TextUtils;

import java.util.HashMap;
import java.util.Map;

import io.intercom.android.sdk.Intercom;
import io.intercom.android.sdk.logger.IntercomLogger;

/**
 * <p>The Registration object is used for registering identified users with Intercom.</p>
 * <p>This is an example of how to register a user with a user ID</p>
 * <pre>
 * {@code
 * Intercom.client().registerIdentifiedUser(Registration.create().withUserId("12345"));
 * }
 * </pre>
 *
 * <p>You can chain methods on the Registration object if you want to provide more data with the registration.</p>
 *
 * <pre>
 * {@code
 * Map<String, Object> userAttributes = new HashMap<>();
 * userAttributes.put("name", "Admin name");
 *
 * Registration.create()
 *     .withUserId("12345")
 *     .withEmail("test@email.com")
 *     .withUserAttributes(userAttributes);
 * Intercom.client().registerIdentifiedUser(registration);
 * }
 * </pre>
 */
public class Registration {
    private String email = "";
    private String userId = "";
    private Map<String, Object> attributes = new HashMap<>();

    private boolean isValidRegistration = true;

    /**
     * Factory to create a new, empty Registration object.
     *
     * @return a new Registration
     */
    public static Registration create() {
        return new Registration();
    }

    /**
     * Add an email address to this registration.
     *
     * @param email the user's email address
     * @return this registration object
     */
    public Registration withEmail(@NonNull String email) {
        if (TextUtils.isEmpty(email)) {
            IntercomLogger.e("Email cannot be null or empty");
            isValidRegistration = false;
        } else {
            this.email = email;
        }
        return this;
    }

    /**
     * Add a unique user identifier to this registration.
     *
     * @param userId A unique identifier for your user
     * @return this registration object
     */
    public Registration withUserId(@NonNull String userId) {
        if (TextUtils.isEmpty(userId)) {
            IntercomLogger.e("UserId cannot be null or empty");
            isValidRegistration = false;
        } else {
            this.userId = userId;
        }
        return this;
    }

    /**
     * <p>Add a map of user attributes to this registration.</p>
     *
     * <p>For full documentation see {@link Intercom#updateUser(Map)}</p>
     *
     * @param attributes the map of attributes to update the user with
     * @return this registration object
     */
    public Registration withUserAttributes(Map<String, Object> attributes) {
        isValidRegistration = false;
        if (attributes == null) {
            IntercomLogger.e("Registration.withUserAttributes method failed: the attributes Map provided is null");
        } else if (attributes.isEmpty()) {
            IntercomLogger.e("Registration.withUserAttributes method failed: the attributes Map provided is empty");
        } else {
            this.attributes = new HashMap<>(attributes);
            isValidRegistration = true;
        }
        return this;
    }

    public String getEmail() {
        return email;
    }

    public String getUserId() {
        return userId;
    }

    public Map<String, Object> getAttributes() {
        return attributes;
    }

    protected boolean isValidRegistration() {
        return isValidRegistration;
    }
}
