/*
 * Copyright 2016 Game Server Services, Inc. or its affiliates. All Rights
 * Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package io.gs2.project;

import java.util.ArrayList;
import java.util.List;

import io.gs2.model.Region;
import io.gs2.util.EncodingUtil;
import org.apache.http.NameValuePair;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.message.BasicNameValuePair;

import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.JsonNode;
import org.json.JSONObject;

import io.gs2.AbstractGs2Client;
import io.gs2.Gs2Constant;
import io.gs2.model.IGs2Credential;
import io.gs2.project.request.*;
import io.gs2.project.result.*;
import io.gs2.project.model.*;

/**
 * GS2 Project API クライアント
 *
 * @author Game Server Services, Inc.
 *
 */
public class Gs2ProjectRestClient extends AbstractGs2Client<Gs2ProjectRestClient> {

	public static String ENDPOINT = "project";

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 */
	public Gs2ProjectRestClient(IGs2Credential credential) {
		super(credential);
	}

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 * @param region リージョン
	 */
	public Gs2ProjectRestClient(IGs2Credential credential, Region region) {
		super(credential, region);
	}

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 * @param region リージョン
	 */
	public Gs2ProjectRestClient(IGs2Credential credential, String region) {
		super(credential, region);

	}

	/**
	 * アカウントを新規作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateAccountResult createAccount(CreateAccountRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FAccountFunctionHandler.createAccount";
        String email = null;
        if(request.getEmail() != null) {
            email = request.getEmail();
        }
        String fullName = null;
        if(request.getFullName() != null) {
            fullName = request.getFullName();
        }
        String companyName = null;
        if(request.getCompanyName() != null) {
            companyName = request.getCompanyName();
        }
        String password = null;
        if(request.getPassword() != null) {
            password = request.getPassword();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(email != null) {
            _body.put("email", email);
        }
        if(fullName != null) {
            _body.put("fullName", fullName);
        }
        if(companyName != null) {
            _body.put("companyName", companyName);
        }
        if(password != null) {
            _body.put("password", password);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateAccountResult.class);
    }

	/**
	 * GS2アカウントを有効化します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public VerifyResult verify(VerifyRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FAccountFunctionHandler.verify";
        String verifyToken = null;
        if(request.getVerifyToken() != null) {
            verifyToken = request.getVerifyToken();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(verifyToken != null) {
            _body.put("verifyToken", verifyToken);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, VerifyResult.class);
    }

	/**
	 * サインインします<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public SignInResult signIn(SignInRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FAccountFunctionHandler.signIn";
        String email = null;
        if(request.getEmail() != null) {
            email = request.getEmail();
        }
        String password = null;
        if(request.getPassword() != null) {
            password = request.getPassword();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(email != null) {
            _body.put("email", email);
        }
        if(password != null) {
            _body.put("password", password);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, SignInResult.class);
    }

	/**
	 * 指定したアカウント名のアカウントトークンを発行<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public IssueAccountTokenResult issueAccountToken(IssueAccountTokenRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FAccountFunctionHandler.issueAccountToken";
        String accountName = null;
        if(request.getAccountName() != null) {
            accountName = request.getAccountName();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(accountName != null) {
            _body.put("accountName", accountName);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, IssueAccountTokenResult.class);
    }

	/**
	 * パスワード再発行トークンを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public ForgetResult forget(ForgetRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FAccountFunctionHandler.forget";
        String email = null;
        if(request.getEmail() != null) {
            email = request.getEmail();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(email != null) {
            _body.put("email", email);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, ForgetResult.class);
    }

	/**
	 * パスワードを再発行<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public IssuePasswordResult issuePassword(IssuePasswordRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FAccountFunctionHandler.issuePassword";
        String issuePasswordToken = null;
        if(request.getIssuePasswordToken() != null) {
            issuePasswordToken = request.getIssuePasswordToken();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(issuePasswordToken != null) {
            _body.put("issuePasswordToken", issuePasswordToken);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, IssuePasswordResult.class);
    }

	/**
	 * GS2アカウントを更新します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateAccountResult updateAccount(UpdateAccountRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FAccountFunctionHandler.updateAccount";
        String email = null;
        if(request.getEmail() != null) {
            email = request.getEmail();
        }
        String fullName = null;
        if(request.getFullName() != null) {
            fullName = request.getFullName();
        }
        String companyName = null;
        if(request.getCompanyName() != null) {
            companyName = request.getCompanyName();
        }
        String password = null;
        if(request.getPassword() != null) {
            password = request.getPassword();
        }
        String accountToken = null;
        if(request.getAccountToken() != null) {
            accountToken = request.getAccountToken();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(email != null) {
            _body.put("email", email);
        }
        if(fullName != null) {
            _body.put("fullName", fullName);
        }
        if(companyName != null) {
            _body.put("companyName", companyName);
        }
        if(password != null) {
            _body.put("password", password);
        }
        if(accountToken != null) {
            _body.put("accountToken", accountToken);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateAccountResult.class);
    }

	/**
	 * GS2アカウントを削除します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteAccountResult deleteAccount(DeleteAccountRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FAccountFunctionHandler.deleteAccount";
        String accountName = null;
        if(request.getAccountName() != null) {
            accountName = request.getAccountName();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteAccountResult.class);
    }

	/**
	 * プロジェクトの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeProjectsResult describeProjects(DescribeProjectsRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FProjectFunctionHandler.describeProjects";
        String accountToken = String.valueOf(request.getAccountToken());
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(accountToken != null) {
            _queryString.add(new BasicNameValuePair("accountToken", String.valueOf(accountToken)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeProjectsResult.class);
    }

	/**
	 * プロジェクトを新規作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateProjectResult createProject(CreateProjectRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FProjectFunctionHandler.createProject";
        String accountToken = null;
        if(request.getAccountToken() != null) {
            accountToken = request.getAccountToken();
        }
        String name = null;
        if(request.getName() != null) {
            name = request.getName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(accountToken != null) {
            _body.put("accountToken", accountToken);
        }
        if(name != null) {
            _body.put("name", name);
        }
        if(description != null) {
            _body.put("description", description);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateProjectResult.class);
    }

	/**
	 * プロジェクトを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetProjectResult getProject(GetProjectRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FProjectFunctionHandler.getProject";
        String projectName = null;
        if(request.getProjectName() != null) {
            projectName = request.getProjectName();
        }
        String accountToken = String.valueOf(request.getAccountToken());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(accountToken != null) {
            _queryString.add(new BasicNameValuePair("accountToken", String.valueOf(accountToken)));
        }
        if(projectName != null) {
            _queryString.add(new BasicNameValuePair("projectName", String.valueOf(projectName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetProjectResult.class);
    }

	/**
	 * プロジェクトトークンを発行します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetProjectTokenResult getProjectToken(GetProjectTokenRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FProjectFunctionHandler.getProjectToken";
        String projectName = null;
        if(request.getProjectName() != null) {
            projectName = request.getProjectName();
        }
        String accountToken = null;
        if(request.getAccountToken() != null) {
            accountToken = request.getAccountToken();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(projectName != null) {
            _queryString.add(new BasicNameValuePair("projectName", String.valueOf(projectName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(accountToken != null) {
            _body.put("accountToken", accountToken);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetProjectTokenResult.class);
    }

	/**
	 * プロジェクトを更新<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateProjectResult updateProject(UpdateProjectRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FProjectFunctionHandler.updateProject";
        String accountToken = null;
        if(request.getAccountToken() != null) {
            accountToken = request.getAccountToken();
        }
        String projectName = null;
        if(request.getProjectName() != null) {
            projectName = request.getProjectName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(projectName != null) {
            _queryString.add(new BasicNameValuePair("projectName", String.valueOf(projectName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(accountToken != null) {
            _body.put("accountToken", accountToken);
        }
        if(description != null) {
            _body.put("description", description);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateProjectResult.class);
    }

	/**
	 * プロジェクトを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteProjectResult deleteProject(DeleteProjectRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/project-handler?handler=gs2_project%2Fhandler%2FProjectFunctionHandler.deleteProject";
        String projectName = null;
        if(request.getProjectName() != null) {
            projectName = request.getProjectName();
        }
        String accountToken = String.valueOf(request.getAccountToken());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(accountToken != null) {
            _queryString.add(new BasicNameValuePair("accountToken", String.valueOf(accountToken)));
        }
        if(projectName != null) {
            _queryString.add(new BasicNameValuePair("projectName", String.valueOf(projectName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteProjectResult.class);
    }
}