/*
 * Decompiled with CFR 0.152.
 */
package io.getlime.security.powerauth.crypto.lib.generator;

import com.google.common.io.BaseEncoding;
import io.getlime.security.powerauth.crypto.lib.encryptor.ecies.kdf.KdfX9_63;
import io.getlime.security.powerauth.crypto.lib.generator.KeyGenerator;
import io.getlime.security.powerauth.crypto.lib.model.RecoveryInfo;
import io.getlime.security.powerauth.crypto.lib.model.RecoverySeed;
import io.getlime.security.powerauth.crypto.lib.model.exception.CryptoProviderException;
import io.getlime.security.powerauth.crypto.lib.model.exception.GenericCryptoException;
import io.getlime.security.powerauth.crypto.lib.util.CRC16;
import io.getlime.security.powerauth.crypto.lib.util.KeyConvertor;
import java.nio.ByteBuffer;
import java.security.InvalidKeyException;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.UUID;
import javax.crypto.SecretKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IdentifierGenerator {
    private static final Logger logger = LoggerFactory.getLogger(IdentifierGenerator.class);
    private static final int BASE32_KEY_LENGTH = 5;
    private static final int ACTIVATION_CODE_BYTES_LENGTH = 12;
    private static final int ACTIVATION_CODE_RANDOM_BYTES_LENGTH = 10;
    private static final int PUK_DERIVATION_MAX_ATTEMPTS = 20;
    private final KeyGenerator keyGenerator = new KeyGenerator();
    private final KeyConvertor keyConvertor = new KeyConvertor();

    public String generateActivationId() {
        return UUID.randomUUID().toString();
    }

    private String generateBase32Token() throws CryptoProviderException {
        byte[] randomBytes = this.keyGenerator.generateRandomBytes(5);
        return BaseEncoding.base32().omitPadding().encode(randomBytes).substring(0, 5);
    }

    public String generateActivationCode() throws CryptoProviderException {
        try {
            byte[] randomBytes = this.keyGenerator.generateRandomBytes(10);
            return this.generateActivationCode(randomBytes);
        }
        catch (GenericCryptoException ex) {
            logger.warn(ex.getMessage(), (Throwable)ex);
            return null;
        }
    }

    public String generateActivationCode(byte[] randomBytes) throws GenericCryptoException {
        if (randomBytes == null || randomBytes.length != 10) {
            throw new GenericCryptoException("Invalid request in generateActivationCode");
        }
        ByteBuffer byteBuffer = ByteBuffer.allocate(12);
        byteBuffer.put(randomBytes);
        CRC16 crc16 = new CRC16();
        crc16.update(randomBytes, 0, 10);
        long crc = crc16.getValue();
        byteBuffer.putShort((short)crc);
        return this.encodeActivationCode(byteBuffer.array());
    }

    public boolean validateActivationCode(String activationCode) {
        if (activationCode == null) {
            return false;
        }
        String partRegexp = "[A-Z2-7]{5}";
        String activationCodeRegexp = partRegexp + "-" + partRegexp + "-" + partRegexp + "-" + partRegexp;
        if (!activationCode.matches(activationCodeRegexp)) {
            return false;
        }
        byte[] activationCodeBytes = BaseEncoding.base32().decode((CharSequence)activationCode.replace("-", ""));
        if (activationCodeBytes.length != 12) {
            return false;
        }
        CRC16 crc16 = new CRC16();
        crc16.update(activationCodeBytes, 0, 10);
        long actualChecksum = crc16.getValue();
        long expectedChecksum = ((long)activationCodeBytes[10] & 0xFFL) << 8 | (long)activationCodeBytes[11] & 0xFFL;
        return expectedChecksum == actualChecksum;
    }

    public RecoveryInfo generateRecoveryCode() throws GenericCryptoException, CryptoProviderException, InvalidKeyException {
        SecretKey secretKey = this.keyGenerator.generateRandomSecretKey();
        return this.generateRecoveryCode(secretKey, 1, false);
    }

    public RecoveryInfo generateRecoveryCode(SecretKey secretKey, int pukCount, boolean exportSeed) throws GenericCryptoException, CryptoProviderException, InvalidKeyException {
        if (secretKey == null) {
            throw new GenericCryptoException("Invalid key");
        }
        byte[] secretKeyBytes = this.keyConvertor.convertSharedSecretKeyToBytes(secretKey);
        byte[] nonceBytes = this.keyGenerator.generateRandomBytes(32);
        byte[] derivedKeyBytes = KdfX9_63.derive(secretKeyBytes, nonceBytes, 26);
        String recoveryCode = this.generateRecoveryCode(derivedKeyBytes);
        SecretKey recoveryPukBaseKey = this.generatePukBaseKey(derivedKeyBytes);
        LinkedHashMap<Integer, Long> pukDerivationIndexes = new LinkedHashMap<Integer, Long>();
        LinkedHashMap<Integer, String> puks = new LinkedHashMap<Integer, String>();
        for (int i = 1; i <= pukCount; ++i) {
            Long derivationIndex;
            String derivedPuk;
            int derivationAttempt = 0;
            do {
                byte[] derivationIndexBytes = this.keyGenerator.generateRandomBytes(8);
                derivationIndex = ByteBuffer.wrap(derivationIndexBytes).getLong();
                derivedPuk = this.generatePuk(recoveryPukBaseKey, derivationIndexBytes);
                if (++derivationAttempt != 20) continue;
                throw new GenericCryptoException("PUK derivation failed due to exceeding maximum number of attempts for generating unique PUK");
            } while (puks.containsValue(derivedPuk));
            puks.put(i, derivedPuk);
            pukDerivationIndexes.put(i, derivationIndex);
        }
        if (exportSeed) {
            RecoverySeed seed = new RecoverySeed(nonceBytes, pukDerivationIndexes);
            return new RecoveryInfo(recoveryCode, puks, seed);
        }
        return new RecoveryInfo(recoveryCode, puks);
    }

    public RecoveryInfo deriveRecoveryCode(SecretKey secretKey, RecoverySeed seed) throws GenericCryptoException, CryptoProviderException, InvalidKeyException {
        if (secretKey == null || seed == null) {
            throw new GenericCryptoException("Invalid input data");
        }
        byte[] secretKeyBytes = this.keyConvertor.convertSharedSecretKeyToBytes(secretKey);
        byte[] nonceBytes = seed.getNonce();
        Map<Integer, Long> pukDerivationIndexes = seed.getPukDerivationIndexes();
        if (nonceBytes == null || pukDerivationIndexes == null || pukDerivationIndexes.isEmpty()) {
            throw new GenericCryptoException("Invalid input data");
        }
        byte[] derivedKeyBytes = KdfX9_63.derive(secretKeyBytes, nonceBytes, 26);
        String recoveryCode = this.generateRecoveryCode(derivedKeyBytes);
        SecretKey recoveryPukBaseKey = this.generatePukBaseKey(derivedKeyBytes);
        LinkedHashMap<Integer, String> puks = new LinkedHashMap<Integer, String>();
        for (int i = 1; i <= pukDerivationIndexes.size(); ++i) {
            Long index = pukDerivationIndexes.get(i);
            byte[] indexBytes = ByteBuffer.allocate(8).putLong(index).array();
            String pukDerived = this.generatePuk(recoveryPukBaseKey, indexBytes);
            puks.put(i, pukDerived);
        }
        RecoveryInfo result = new RecoveryInfo();
        result.setRecoveryCode(recoveryCode);
        result.setPuks(puks);
        return result;
    }

    private String generateRecoveryCode(byte[] derivedKeyBytes) throws GenericCryptoException {
        byte[] recoveryCodeBytes = new byte[10];
        System.arraycopy(derivedKeyBytes, 0, recoveryCodeBytes, 0, 10);
        return this.generateActivationCode(recoveryCodeBytes);
    }

    private SecretKey generatePukBaseKey(byte[] derivedKeyBytes) {
        byte[] recoveryPukBaseKeyBytes = new byte[16];
        System.arraycopy(derivedKeyBytes, 10, recoveryPukBaseKeyBytes, 0, 16);
        return this.keyConvertor.convertBytesToSharedSecretKey(recoveryPukBaseKeyBytes);
    }

    private String generatePuk(SecretKey recoveryPukBaseKey, byte[] indexBytes) throws CryptoProviderException, InvalidKeyException, GenericCryptoException {
        SecretKey pukKey = this.keyGenerator.deriveSecretKey(recoveryPukBaseKey, indexBytes);
        byte[] pukKeyBytes = this.keyConvertor.convertSharedSecretKeyToBytes(pukKey);
        byte[] truncatedBytes = new byte[8];
        System.arraycopy(pukKeyBytes, 8, truncatedBytes, 0, 8);
        long puk = (ByteBuffer.wrap(truncatedBytes).getLong() & 0xFFFFFFFFFFL) % (long)Math.pow(10.0, 10.0);
        return String.format("%010d", puk);
    }

    private String encodeActivationCode(byte[] activationCodeBytes) {
        String base32Encoded = BaseEncoding.base32().omitPadding().encode(activationCodeBytes);
        return base32Encoded.substring(0, 5) + "-" + base32Encoded.substring(5, 10) + "-" + base32Encoded.substring(10, 15) + "-" + base32Encoded.substring(15, 20);
    }
}

