/*
 * Decompiled with CFR 0.152.
 */
package io.getlime.security.powerauth.crypto.lib.encryptor.ecies;

import com.google.common.primitives.Bytes;
import io.getlime.security.powerauth.crypto.lib.encryptor.ecies.EciesEnvelopeKey;
import io.getlime.security.powerauth.crypto.lib.encryptor.ecies.exception.EciesException;
import io.getlime.security.powerauth.crypto.lib.encryptor.ecies.model.EciesCryptogram;
import io.getlime.security.powerauth.crypto.lib.generator.KeyGenerator;
import io.getlime.security.powerauth.crypto.lib.model.exception.CryptoProviderException;
import io.getlime.security.powerauth.crypto.lib.model.exception.GenericCryptoException;
import io.getlime.security.powerauth.crypto.lib.util.AESEncryptionUtils;
import io.getlime.security.powerauth.crypto.lib.util.HMACHashUtilities;
import io.getlime.security.powerauth.crypto.lib.util.KeyConvertor;
import java.security.InvalidKeyException;
import java.security.PublicKey;
import java.security.interfaces.ECPublicKey;
import java.util.Arrays;
import javax.crypto.SecretKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EciesEncryptor {
    private static final Logger logger = LoggerFactory.getLogger(EciesEncryptor.class);
    private final AESEncryptionUtils aes = new AESEncryptionUtils();
    private final HMACHashUtilities hmac = new HMACHashUtilities();
    private final KeyConvertor keyConvertor = new KeyConvertor();
    private final KeyGenerator keyGenerator = new KeyGenerator();
    private final PublicKey publicKey;
    private final byte[] sharedInfo1;
    private final byte[] sharedInfo2;
    private EciesEnvelopeKey envelopeKey;
    private boolean canEncryptData;
    private boolean canDecryptData;
    private byte[] ivForDecryption;

    public EciesEncryptor(ECPublicKey publicKey) {
        this(publicKey, null, null);
    }

    public EciesEncryptor(ECPublicKey publicKey, byte[] sharedInfo1, byte[] sharedInfo2) {
        this.publicKey = publicKey;
        this.sharedInfo1 = sharedInfo1;
        this.sharedInfo2 = sharedInfo2;
        this.canEncryptData = true;
        this.canDecryptData = false;
    }

    public EciesEncryptor(EciesEnvelopeKey envelopeKey, byte[] sharedInfo2) {
        this.publicKey = null;
        this.envelopeKey = envelopeKey;
        this.sharedInfo1 = null;
        this.sharedInfo2 = sharedInfo2;
        this.canEncryptData = false;
        this.canDecryptData = true;
    }

    public EciesCryptogram encryptRequest(byte[] data, boolean useIv) throws EciesException {
        if (data == null) {
            throw new EciesException("Parameter data for request encryption is null");
        }
        if (!this.canEncryptRequest()) {
            throw new EciesException("Request encryption is not allowed");
        }
        this.envelopeKey = EciesEnvelopeKey.fromPublicKey(this.publicKey, this.sharedInfo1);
        return this.encrypt(data, useIv);
    }

    public byte[] decryptResponse(EciesCryptogram cryptogram) throws EciesException {
        if (cryptogram == null || cryptogram.getEncryptedData() == null || cryptogram.getMac() == null) {
            throw new EciesException("Parameter cryptogram for response decryption is invalid");
        }
        if (!this.canDecryptResponse()) {
            throw new EciesException("Response decryption is not allowed");
        }
        return this.decrypt(cryptogram);
    }

    public byte[] getSharedInfo2() {
        return this.sharedInfo2;
    }

    public EciesEnvelopeKey getEnvelopeKey() {
        return this.envelopeKey;
    }

    private boolean canEncryptRequest() {
        return this.canEncryptData && this.publicKey != null;
    }

    private boolean canDecryptResponse() {
        return this.canDecryptData && this.envelopeKey.isValid() && this.ivForDecryption != null;
    }

    private EciesCryptogram encrypt(byte[] data, boolean useIv) throws EciesException {
        try {
            byte[] iv;
            byte[] nonce;
            if (useIv) {
                nonce = this.keyGenerator.generateRandomBytes(16);
                iv = this.envelopeKey.deriveIvForNonce(nonce);
            } else {
                nonce = null;
                iv = new byte[16];
            }
            byte[] encKeyBytes = this.envelopeKey.getEncKey();
            SecretKey encKey = this.keyConvertor.convertBytesToSharedSecretKey(encKeyBytes);
            byte[] encryptedData = this.aes.encrypt(data, iv, encKey);
            byte[] macData = this.sharedInfo2 == null ? encryptedData : Bytes.concat((byte[][])new byte[][]{encryptedData, this.sharedInfo2});
            byte[] mac = this.hmac.hash(this.envelopeKey.getMacKey(), macData);
            this.canEncryptData = false;
            this.canDecryptData = true;
            this.ivForDecryption = iv;
            return new EciesCryptogram(this.envelopeKey.getEphemeralKeyPublic(), mac, encryptedData, nonce);
        }
        catch (CryptoProviderException | GenericCryptoException | InvalidKeyException ex) {
            logger.warn(ex.getMessage(), (Throwable)ex);
            throw new EciesException("Request encryption failed", ex);
        }
    }

    private byte[] decrypt(EciesCryptogram cryptogram) throws EciesException {
        try {
            byte[] macData = this.sharedInfo2 == null ? cryptogram.getEncryptedData() : Bytes.concat((byte[][])new byte[][]{cryptogram.getEncryptedData(), this.sharedInfo2});
            byte[] mac = this.hmac.hash(this.envelopeKey.getMacKey(), macData);
            if (!Arrays.equals(mac, cryptogram.getMac())) {
                throw new EciesException("Invalid MAC");
            }
            byte[] encKeyBytes = this.envelopeKey.getEncKey();
            SecretKey encKey = this.keyConvertor.convertBytesToSharedSecretKey(encKeyBytes);
            byte[] iv = this.ivForDecryption;
            this.canDecryptData = false;
            this.ivForDecryption = null;
            return this.aes.decrypt(cryptogram.getEncryptedData(), iv, encKey);
        }
        catch (CryptoProviderException | GenericCryptoException | InvalidKeyException ex) {
            logger.warn(ex.getMessage(), (Throwable)ex);
            throw new EciesException("Response decryption failed", ex);
        }
    }
}

