/*
 * Decompiled with CFR 0.152.
 */
package io.getlime.security.powerauth.crypto.lib.encryptor.ecies;

import com.google.common.primitives.Bytes;
import io.getlime.security.powerauth.crypto.lib.config.PowerAuthConfiguration;
import io.getlime.security.powerauth.crypto.lib.encryptor.ecies.exception.EciesException;
import io.getlime.security.powerauth.crypto.lib.encryptor.ecies.kdf.KdfX9_63;
import io.getlime.security.powerauth.crypto.lib.generator.KeyGenerator;
import io.getlime.security.powerauth.crypto.lib.model.exception.GenericCryptoException;
import io.getlime.security.powerauth.crypto.lib.util.HMACHashUtilities;
import io.getlime.security.powerauth.provider.CryptoProviderUtil;
import io.getlime.security.powerauth.provider.exception.CryptoProviderException;
import java.nio.ByteBuffer;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.InvalidKeySpecException;
import javax.crypto.SecretKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EciesEnvelopeKey {
    private static final Logger logger = LoggerFactory.getLogger(EciesEnvelopeKey.class);
    private static final int ENC_KEY_OFFSET = 0;
    private static final int ENC_KEY_SIZE = 16;
    private static final int MAC_KEY_OFFSET = 16;
    private static final int MAC_KEY_SIZE = 16;
    private static final int IV_KEY_OFFSET = 32;
    private static final int IV_KEY_SIZE = 16;
    private static final int NONCE_SIZE = 16;
    private static final int ENVELOPE_KEY_SIZE = 48;
    private static final CryptoProviderUtil keyConverter = PowerAuthConfiguration.INSTANCE.getKeyConvertor();
    private static final KeyGenerator keyGenerator = new KeyGenerator();
    private static final HMACHashUtilities hmac = new HMACHashUtilities();
    private final byte[] secretKey;
    private final byte[] ephemeralKeyPublic;

    public EciesEnvelopeKey(byte[] secretKey, byte[] ephemeralPublicKey) {
        this.secretKey = secretKey;
        this.ephemeralKeyPublic = ephemeralPublicKey;
    }

    static EciesEnvelopeKey fromPublicKey(PublicKey publicKey, byte[] sharedInfo1) throws EciesException {
        try {
            KeyPair ephemeralKeyPair = keyGenerator.generateKeyPair();
            PrivateKey ephemeralPrivateKey = ephemeralKeyPair.getPrivate();
            PublicKey ephemeralPublicKey = ephemeralKeyPair.getPublic();
            byte[] ephemeralPublicKeyBytes = keyConverter.convertPublicKeyToBytes(ephemeralPublicKey);
            SecretKey ephemeralSecretKey = keyGenerator.computeSharedKey(ephemeralPrivateKey, publicKey, true);
            byte[] info1Data = sharedInfo1 == null ? ephemeralPublicKeyBytes : Bytes.concat((byte[][])new byte[][]{sharedInfo1, ephemeralPublicKeyBytes});
            byte[] secretKey = KdfX9_63.derive(keyConverter.convertSharedSecretKeyToBytes(ephemeralSecretKey), info1Data, 48);
            return new EciesEnvelopeKey(secretKey, ephemeralPublicKeyBytes);
        }
        catch (GenericCryptoException | CryptoProviderException | InvalidKeyException ex) {
            logger.warn(ex.getMessage(), ex);
            throw new EciesException("Key derivation failed", ex);
        }
    }

    static EciesEnvelopeKey fromPrivateKey(PrivateKey ephemeralKeyPrivate, byte[] ephemeralPublicKeyBytes, byte[] sharedInfo1) throws EciesException {
        try {
            PublicKey ephemeralPublicKey = keyConverter.convertBytesToPublicKey(ephemeralPublicKeyBytes);
            SecretKey ephemeralSecretKey = keyGenerator.computeSharedKey(ephemeralKeyPrivate, ephemeralPublicKey, true);
            byte[] info1Data = sharedInfo1 == null ? ephemeralPublicKeyBytes : Bytes.concat((byte[][])new byte[][]{sharedInfo1, ephemeralPublicKeyBytes});
            byte[] secretKey = KdfX9_63.derive(keyConverter.convertSharedSecretKeyToBytes(ephemeralSecretKey), info1Data, 48);
            return new EciesEnvelopeKey(secretKey, ephemeralPublicKeyBytes);
        }
        catch (GenericCryptoException | CryptoProviderException | InvalidKeyException | InvalidKeySpecException ex) {
            logger.warn(ex.getMessage(), ex);
            throw new EciesException("Key derivation failed", ex);
        }
    }

    public byte[] getEncKey() throws EciesException {
        if (!this.isValid()) {
            throw new EciesException("Encryption key is not valid");
        }
        ByteBuffer byteBuffer = ByteBuffer.allocate(16);
        byteBuffer.put(this.secretKey, 0, 16);
        return byteBuffer.array();
    }

    public byte[] getMacKey() throws EciesException {
        if (!this.isValid()) {
            throw new EciesException("MAC key is not valid");
        }
        ByteBuffer byteBuffer = ByteBuffer.allocate(16);
        byteBuffer.put(this.secretKey, 16, 16);
        return byteBuffer.array();
    }

    public byte[] getIvKey() throws EciesException {
        if (!this.isValid()) {
            throw new EciesException("IV key is not valid");
        }
        ByteBuffer byteBuffer = ByteBuffer.allocate(16);
        byteBuffer.put(this.secretKey, 32, 16);
        return byteBuffer.array();
    }

    public byte[] deriveIvForNonce(byte[] nonce) throws EciesException {
        if (nonce == null) {
            throw new EciesException("Nonce for IV derivation is missing");
        }
        if (nonce.length != 16) {
            throw new EciesException("Nonce for IV derivation is not valid");
        }
        try {
            return keyGenerator.convert32Bto16B(hmac.hash(this.getIvKey(), nonce));
        }
        catch (GenericCryptoException | CryptoProviderException ex) {
            logger.warn(ex.getMessage(), ex);
            throw new EciesException("IV derivation failed", ex);
        }
    }

    public byte[] getSecretKey() throws EciesException {
        if (!this.isValid()) {
            throw new EciesException("Secret key is not valid");
        }
        return this.secretKey;
    }

    public byte[] getEphemeralKeyPublic() {
        return this.ephemeralKeyPublic;
    }

    public boolean isValid() {
        return this.secretKey.length == 48;
    }
}

