package cronapp.framework.authentication.token;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import cronapi.AppConfig;
import cronapi.RestClient;
import cronapi.Var;
import cronapi.database.DatabaseQueryManager;
import cronapi.database.HistoryListener;
import cronapi.database.TransactionManager;
import cronapi.util.Operations;
import cronapp.framework.LockedUserException;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.api.User;
import cronapp.framework.api.response.DefaultResponse;
import cronapp.framework.authentication.external.ExternalAuthenticationConfig;
import cronapp.framework.authentication.security.CronappUserDetails;
import cronapp.framework.authentication.social.SocialConfig;
import cronapp.framework.authentication.token.google.CaptchaVerify;
import cronapp.framework.authentication.token.google.ICaptchaVerify;
import cronapp.framework.core.CronappConfiguration;
import cronapp.framework.core.CronappSettingsService;
import cronapp.framework.i18n.Messages;
import cronapp.framework.persistence.PasswordConstraintException;
import cronapp.framework.tenant.TenantComponent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.Nullable;
import org.springframework.mobile.device.Device;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.AuthorityUtils;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.encrypt.Encryptors;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.stream.Collectors;

import static cronapp.framework.authentication.external.ExternalAuthenticationConfig.*;

@RestController
@RequestMapping("auth")
public class AuthenticationController {

  private final Logger logger = LoggerFactory.getLogger(getClass());
  private static final String SCOPE_UPDATE_CURRENT_USER_PASSWORD = "update:current_user:password";

  private TenantComponent tenantComponent;

  private ICaptchaVerify captchaVerify;

  public AuthenticationController(@Nullable TenantComponent tenantComponent) {
    this.captchaVerify = new CaptchaVerify();
    this.tenantComponent = tenantComponent;
  }

  @RequestMapping(method = RequestMethod.POST)
  public ResponseEntity<AuthenticationResponse> authenticationRequest(@RequestParam String username, String password, Device device, @RequestHeader(name = "X-AUTH-TOKEN", required = false) String token, final HttpServletRequest request) throws AuthenticationException {
    return auth(username, password, device, "local", token, null, request);
  }

  public ResponseEntity<AuthenticationResponse> auth(String username, String password, Device device, String provider, String authToken, JsonObject details, final HttpServletRequest request) throws AuthenticationException {

    if (authToken != null) {
      String localProvider = TokenUtils.getProviderFromToken(authToken);
      if (provider != null && !"local".equals(localProvider) && username.equals("#OAUTH#")) {
        username = TokenUtils.getUsernameFromToken(authToken);
        provider = localProvider;
      }
    }

    boolean isOauth = !"local".equals(provider);
    boolean autoSignUp = false;
    boolean externalAuthenticationSuccess = false;

    Authentication externalToken;
    CronappUserDetails externalUserDetails = null;

    if (isExternalAuth()) {
      externalToken = authenticateExternally(new UsernamePasswordAuthenticationToken(username, password));
      provider = getExternalAuthType();
      externalAuthenticationSuccess = true;
      autoSignUp = AppConfig.autoSignUp();

      if (externalToken.getPrincipal() instanceof CronappUserDetails) {
        externalUserDetails = (CronappUserDetails) externalToken.getPrincipal();
      }
    } else if (ExternalAuthenticationConfig.isSocial(provider)) {
      autoSignUp = SocialConfig.isAutoSignUp();
    } else if("SSO".equalsIgnoreCase(provider)) {
      // TODO: Validar se essa definição de autoSignUp pode ser aplicada para todos os tipos de autenticação
      autoSignUp = AppConfig.autoSignUp();
    }


    ApiManager apiManager = ApiManager.byUserAndPassword(username, password, provider, autoSignUp, details);

    User userNoPassword = null;
    try {

      userNoPassword = ApiManager.byUser(username).getUser();

      if (ApiManager.isUserLocked(userNoPassword)) {
        this.doLogAuthOperation("Fail", "auth", username, Arrays.asList("Reason", "UserLocked"));
        throw new LockedUserException(Messages.getString("UserLocked"));
      }

      verifyRecaptcha(username, request);

      User user = apiManager.getUser(externalUserDetails);

      if (user == null) {
        this.doLogAuthOperation("Fail","auth", username, Arrays.asList("Reason", "UserNotFound"));
        throw new UsernameNotFoundException(Messages.getString("UserNotFound"));
      }

      if (isExternalAuth() && (!isOauth && !externalAuthenticationSuccess)) {
        this.doLogAuthOperation("Fail","auth", username, Arrays.asList("Reason", "UserOrPassordInvalids"));
        throw new BadCredentialsException(Messages.getString("UserOrPassordInvalids"));
      }

      if (!isExternalAuth() && (!isOauth && !apiManager.passwordMatches(password, user.getPassword()))) {
        this.doLogAuthOperation("Fail","auth", username, Arrays.asList("Reason", "UserOrPassordInvalids"));
        throw new BadCredentialsException(Messages.getString("UserOrPassordInvalids"));
      }

      Collection<? extends GrantedAuthority> authorities;

      if (authToken != null) {
        authorities = TokenUtils.getAuthoritiesFromToken(authToken).stream().map(SimpleGrantedAuthority::new).collect(Collectors.toSet());
      }
      else if (externalUserDetails == null) {
        authorities = apiManager.getAuthorities();
      }
      else {
        authorities = externalUserDetails.getAuthorities();
      }

      if (!AppConfig.isNull(details) && !AppConfig.isNull(details.get("authorities"))) {

        List<String> mergedList = new ArrayList<>();
        authorities.forEach( s -> mergedList.add(s.getAuthority()));

        JsonArray authoritiesList = details.get("authorities").getAsJsonArray();
        authoritiesList.forEach(a -> {
          String role = a.getAsJsonObject().get("role").getAsString();
          if (!mergedList.contains(role)) {
            mergedList.add(role);
          }
        });

        authorities = AuthorityUtils.commaSeparatedStringToAuthorityList(StringUtils.collectionToCommaDelimitedString(mergedList));
      }

      UserDetails userDetails = new org.springframework.security.core.userdetails.User(username, "password", true, true, true, true, authorities);

      SecurityContextHolder.getContext()
          .setAuthentication(new UsernamePasswordAuthenticationToken(userDetails, "password", authorities));

      StringJoiner roles = new StringJoiner(",");

      roles.add("Public");
      roles.add("Authenticated");

      boolean root = false;
      for (GrantedAuthority role : authorities) {
        if (!role.getAuthority().equals("Public") && !role.getAuthority().equals("Authenticated")) {
          roles.add(role.getAuthority());
          if (role.getAuthority().equalsIgnoreCase("Administrators")) {
            root = true;
          }
        }
      }

      if (tenantComponent != null) {
        tenantComponent.authenticationTenant(user.getUsername());
      }

      User userWithoutPass = user.resetPassword();

      if (EventsManager.hasEvent("onLogin") && EventsManager.getEvent("onLogin").get("type").getAsString().equalsIgnoreCase("server")) {
        EventsManager.executeEventOnTransaction("onLogin", Var.valueOf("username", username));
      }

      String token = TokenUtils.generateToken(userDetails, user.getName(), device, provider);
      Date expires = TokenUtils.getExpirationDateFromToken(token);

      ApiManager.unlockUser(user);
      this.doLogAuthOperation("Success", "auth", username, Arrays.asList("Roles", roles.toString()));
      return ResponseEntity.ok(new AuthenticationResponse(userWithoutPass, token, expires.getTime(), roles.toString(), root));
    } catch (Exception e) {
      logger.error(Messages.getString("AuthError", e.getMessage()), e);
      ApiManager.attemptFailed(userNoPassword);

      if (!ApiManager.isUserLocked(userNoPassword) && ApiManager.getFailedAttempts(userNoPassword) > AppConfig.getFailedAttempts()) {
        ApiManager.lockUser(userNoPassword);
        this.doLogAuthOperation("Fail", "auth", username, Arrays.asList("Reason", "UserLocked10Min"));
        throw new AuthenticationServiceException(Messages.getString("UserLocked10Min"));
      }
      this.doLogAuthOperation("Fail", "auth", username, Arrays.asList("Reason", e.getMessage()));
      throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
    }
  }

  @RequestMapping(value = "refresh", method = RequestMethod.GET)
  public ResponseEntity<?> authenticationRequest(HttpServletRequest request) {
    String tokenHeader = TokenUtils.AUTH_HEADER_NAME;
    String token = request.getHeader(tokenHeader);
    Date expires = TokenUtils.getExpirationDateFromToken(token);

    if (!TokenUtils.canTokenBeRefreshed(token, expires) || !TokenUtils.getScopeFromToken(token).isEmpty()) {
      return ResponseEntity.badRequest().body(null);
    }

    String refreshedToken = TokenUtils.refreshToken(token);
    expires = TokenUtils.getExpirationDateFromToken(token);
    String username = TokenUtils.getUsernameFromToken(token);
    String name = TokenUtils.getNameFromToken(token);
    if ((name == null) || ("".equals(name))) {
      name = username;
    }

    ApiManager apiManager = ApiManager.byUser(username);

    StringJoiner roles = new StringJoiner(",");

    roles.add("Public");
    roles.add("Authenticated");

    boolean root = false;

    if (isExternalAuth() || getExternalAuthType().equalsIgnoreCase("SSO")) {
      roles = new StringJoiner(",");
      List<String> authoritiesFromToken = TokenUtils.getAuthoritiesFromToken(token);
      authoritiesFromToken.forEach(roles::add);
      root = authoritiesFromToken.stream().filter(a -> a.equalsIgnoreCase("Administrators")).findFirst().isPresent();
    }
    else {
      for (GrantedAuthority role : apiManager.getAuthorities()) {
        roles.add(role.getAuthority());
        if (role.getAuthority().equalsIgnoreCase("Administrators")) {
          root = true;
        }
      }
    }

    User userWithoutPass = new User(name, username);

    AuthenticationResponse authenticationResponse = new AuthenticationResponse(userWithoutPass, refreshedToken,
        expires.getTime(), roles.toString(), root);

    return ResponseEntity.ok(authenticationResponse);
  }

  private void verifyRecaptcha(String username, HttpServletRequest request) {
    try {
      Assert.isTrue(captchaVerify.processRequest(username, request), "");
    } catch (Exception e) {
      logger.error(Messages.getString("AuthError", e.getMessage()), e);
      throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
    }
  }

  private void doLogAuthOperation(String type, String command, String user, List<String> params){

    try {
      DatabaseQueryManager logManager = HistoryListener.getAuditLogManager();

      if (logManager != null) {
        String className = logManager.getEntity();
        Class<?> c = Class.forName(className);
        TransactionManager.begin(c);
        JsonObject json = new JsonObject();
        JsonArray arrayParams = new JsonArray();
        json.add("parameters", arrayParams);
        for (String p : params) {
          arrayParams.add(p);
        }
        Var auditLog = new Var(new LinkedHashMap<>());
        auditLog.set("type", "app.authorization." + type);
        auditLog.set("command", command);
        auditLog.set("category", "Authorization");
        auditLog.set("date", new Date());
        auditLog.set("objectData", json.toString());
        if (RestClient.getRestClient() != null) {
          auditLog.set("user", user);
          auditLog.set("host", RestClient.getRestClient().getHost());
          auditLog.set("agent", RestClient.getRestClient().getAgent());
        }
        auditLog.set("server", HistoryListener.CURRENT_IP);
        auditLog.set("affectedFields", null);
        auditLog.set("application", AppConfig.guid());
        logManager.insert(auditLog);
        TransactionManager.commit(c);
      }
    }catch (Exception e){
      logger.error("Error on logging: " + e.getMessage());
    }

  }

  @PostMapping("reset-password")
  public ResponseEntity<String> resetPassword(@RequestParam("email") String email) {
    try {
      var user = ApiManager.byUser(email).getUser();
      var claims = new HashMap<String, Object>();
      claims.put("sub", user.getUsername());
      claims.put("scope", SCOPE_UPDATE_CURRENT_USER_PASSWORD);
      var token = TokenUtils.generateToken(claims, TokenUtils.generateExpirationDate());
      Operations.callBlockly(Var.valueOf("UserManager:sendResetPasswordEmail"), Var.valueOf(user.getEmail()), Var.valueOf(user.getName()), Var.valueOf(token));
    } catch (Exception e) {
      logger.error(Messages.getString("AuthError", e.getMessage()), e);
    }

    return new ResponseEntity<>("{}", HttpStatus.OK);
  }

  @PostMapping("confirm-reset-password")
  public void confirmResetPassword(@RequestParam("password") String password, @RequestParam(value = "otp", required = false) String otp, @RequestHeader(TokenUtils.AUTH_HEADER_NAME) String authToken) {
    if (TokenUtils.isTokenExpired(authToken)) {
      throw new ForbiddenException(Messages.getString("ResetPasswordTokenExpired"));
    }

    String username = TokenUtils.getUsernameFromToken(authToken);
    List<String> scopes = TokenUtils.getScopeFromToken(authToken);
    var claims = TokenUtils.getClaimsFromToken(authToken);

    if (username == null || !scopes.contains(SCOPE_UPDATE_CURRENT_USER_PASSWORD)) {
      throw new ForbiddenException(Messages.getString("UserOrPassordInvalids"));
    }
    var settingsService = CronappConfiguration.getBean(CronappSettingsService.class);

    var encryptedOneTimePassword = String.valueOf(claims.get("otp"));
    var decryptedOneTimePassword = Encryptors.text(settingsService.getEncryptionKey(), settingsService.getEncryptionSalt()).decrypt(encryptedOneTimePassword);

    if (encryptedOneTimePassword != null && !decryptedOneTimePassword.equals(otp)) {
      throw new ForbiddenException(Messages.getString("InvalidOTP"));
    }

    if (EventsManager.hasEvent("onResetPassword")) {
      EventsManager.executeEventOnTransaction("onResetPassword", Var.valueOf(username), Var.valueOf(password));
      return;
    }

    ApiManager manager = ApiManager.byUser(username);
    try {
      manager.updatePassword(password);
    } catch (Exception e) {
      PasswordConstraintException passwordConstraintException = PasswordConstraintException.unwrap(e);

      if (passwordConstraintException != null) {
        throw passwordConstraintException;
      }

      throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
    }
  }

  @PostMapping("signup")
  public ResponseEntity<DefaultResponse> signUp(@RequestBody Var fields) {

    DefaultResponse defaultResponse = new DefaultResponse();

	Boolean avaliableSignup = AppConfig.getIfRegistrationAvailable();
    if (!avaliableSignup) {
      return new ResponseEntity<>(defaultResponse.parseResponse(HttpStatus.FORBIDDEN.value(), Messages.getString("UserRegisterNotAvaliable")), HttpStatus.FORBIDDEN);
    }

    Map mapUser = fields.getObjectAsMap();

    if (mapUser.get("name") == null) {
      mapUser.put("name", mapUser.get("username"));
    }

    mapUser.put("normalizedUserName", mapUser.get("username"));
    mapUser.put("normalizedEmail", mapUser.get("email"));

    try {

      User userNoPassword = ApiManager.byUser(mapUser.get("username").toString()).getUser();
      if (userNoPassword != null) {
        return new ResponseEntity<>(defaultResponse.parseResponse(HttpStatus.BAD_REQUEST.value(), Messages.getString("UserAlreadyExists")), HttpStatus.BAD_REQUEST);
      }

      ApiManager.createUser(fields);
    } catch (Exception err) {
      PasswordConstraintException passwordConstraintException = PasswordConstraintException.unwrap(err);

      if (passwordConstraintException != null) {
        throw passwordConstraintException;
      }

      return new ResponseEntity<>(defaultResponse.parseResponse(HttpStatus.FORBIDDEN.value(), Messages.getString("AuthError", err.getMessage())), HttpStatus.FORBIDDEN);
    }

    return new ResponseEntity<>(defaultResponse.parseResponse(HttpStatus.CREATED.value(), Messages.getString("UserRegisteredSuccessfully")), HttpStatus.CREATED);
  }

}
