package cronapp.framework.rest;

import cronapi.AppConfig;
import cronapi.Var;
import cronapp.framework.CannotChangePasswordException;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.persistence.InvalidConfirmationPasswordException;
import cronapp.framework.i18n.Messages;
import cronapp.framework.persistence.PasswordConstraintException;
import org.springframework.http.HttpStatus;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.web.bind.annotation.*;

/**
 * Controller responsável por gerir a troca de
 * senha do usuário através de serviço REST
 *
 * @author Techne
 */
@RestController
@RequestMapping(value = "/changePassword")
public class ChangePassword {

  @ResponseStatus(HttpStatus.OK)
  @PostMapping
  public void post(String oldPassword, String newPassword, String newPasswordConfirmation) throws Exception {
    String authenticationType = AppConfig.type();

    if (authenticationType != null && !authenticationType.equalsIgnoreCase("internal") && !authenticationType.equalsIgnoreCase("token")) {
      throw new CannotChangePasswordException(Messages.getString("CannotChangePassword"));
    }

    if (!newPassword.equals(newPasswordConfirmation)) {
      throw new InvalidConfirmationPasswordException();
    }

    String username = AuthenticationUtil.getUsername();

    if (EventsManager.hasEvent("onChangePassword")) {
      EventsManager.executeEventOnTransaction("onChangePassword", Var.valueOf("username", username), Var.valueOf("oldPassword", oldPassword), Var.valueOf("newPassword", newPassword), Var.valueOf("newPasswordConfirmation", newPasswordConfirmation));
    } else {
      ApiManager apiManager = ApiManager.byUser(username);
      cronapp.framework.api.User user = apiManager.getUser();
      if (user == null) {
        throw new UsernameNotFoundException(Messages.getString("UserNotFound"));
      }
      String password = user.getPassword();
      if (!apiManager.passwordMatches(oldPassword, password)) {
        throw new BadCredentialsException(Messages.getString("UserOrPassordInvalids"));
      }

      try {
        apiManager.updatePassword(newPassword);
      } catch (Exception e) {
        PasswordConstraintException passwordConstraintException = PasswordConstraintException.unwrap(e);

        if (passwordConstraintException != null) {
          throw passwordConstraintException;
        }

        throw e;
      }
    }
  }
}