package cronapp.framework.persistence;

import cronapp.framework.CronappFrameworkException;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.persistence.descriptors.DescriptorEvent;
import org.eclipse.persistence.descriptors.DescriptorEventAdapter;
import org.eclipse.persistence.internal.helper.DatabaseField;
import org.eclipse.persistence.internal.sessions.AbstractSession;
import org.eclipse.persistence.mappings.DatabaseMapping;
import org.eclipse.persistence.sessions.Record;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.stream.Collectors;

public class PasswordEncoderListener extends DescriptorEventAdapter {

  @Override
  public void aboutToInsert(DescriptorEvent event) {
    super.aboutToInsert(event);
    hashPassword(event);
  }

  @Override
  public void aboutToUpdate(DescriptorEvent event) {
    super.aboutToUpdate(event);
    hashPassword(event);
  }

  @SuppressWarnings("unchecked")
  private <T> T getProperty(Object bean, String name) {
    try {
      Object propertyvalue = PropertyUtils.getProperty(bean, name);
      return (T) propertyvalue;
    } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
      throw new CronappFrameworkException(String.format("Error getting %s property value", name), e);
    }
  }

  private void hashPassword(DescriptorEvent event) {
    for (ValidPassword validPassword : event.getObject().getClass().getAnnotationsByType(ValidPassword.class)) {
      hashPassword(event, validPassword.passwordProperty(), validPassword.passwordHistoryProperty());
    }
  }

  private DatabaseField getDatabaseField(DescriptorEvent event, String attributeName) {
    DatabaseMapping databaseMapping = event.getDescriptor().getMappingForAttributeName(attributeName);
    return databaseMapping == null
        ? null
        : databaseMapping.getField();
  }

  @SuppressWarnings("unchecked")
  private void hashPassword(DescriptorEvent event, String passwordProperty, String passwordHistoryProperty) {
    Object sourceBean = event.getSource();
    Object originalBean = event.getOriginalObject();

    String sourcePassword = getProperty(sourceBean, passwordProperty);
    String sourcePasswordHistory = getProperty(sourceBean, passwordHistoryProperty);

    String originalPassword = getProperty(originalBean, passwordProperty);
    String originalPasswordHistory = getProperty(originalBean, passwordProperty);

    if (sourcePassword == null) {
      return;
    }

    DatabaseField passwordField = getDatabaseField(event, passwordProperty);
    DatabaseField passwordHistoryField = getDatabaseField(event, passwordHistoryProperty);

    String encodedPassword = PasswordEncoder.getPasswordEncoder().isEncoded(sourcePassword)
        ? sourcePassword
        : PasswordEncoder.getPasswordEncoder().encode(sourcePassword);

    String encodedPasswordHistory;
    if (sourcePasswordHistory != null && sourcePasswordHistory.contains(encodedPassword)) {
      encodedPasswordHistory = sourcePasswordHistory;
    } else {
      encodedPasswordHistory = StringUtils.isBlank(sourcePasswordHistory)
          ? encodedPassword
          : encodedPassword + "," + sourcePasswordHistory;
    }

    Record record = event.getRecord();

    if (passwordField != null) {
      if (StringUtils.equals(originalPassword, encodedPassword)) {
        record.remove(passwordField);
      } else {
        record.put(passwordField, encodedPassword);
      }
    }

    if (passwordHistoryField != null) {
      if (StringUtils.equals(originalPasswordHistory, encodedPasswordHistory)) {
        record.remove(passwordHistoryField);
      } else {
        var truncatedPasswordHistory = List.of(encodedPasswordHistory.split(",")).stream()
            .limit(3)
            .collect(Collectors.joining(","));
        record.put(passwordHistoryField, truncatedPasswordHistory);
      }
    }

    AbstractSession rootSession = event.getSession().getRootSession(event.getQuery());
    Object primaryKey = event.getSession().getId(sourceBean);
    Class<?> beanClass = sourceBean.getClass();
    rootSession.getIdentityMapAccessor().invalidateObject(primaryKey, beanClass);
  }
}
