package cronapp.framework.persistence;

import cronapi.AppConfig;
import cronapp.framework.CronappFrameworkException;
import cronapp.framework.i18n.Messages;
import org.apache.commons.beanutils.PropertyUtils;
import org.passay.*;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;


public class PasswordConstraintValidator implements ConstraintValidator<ValidPassword, Object> {
  private String passwordProperty;

  private String passwordHistoryProperty;

  @Override
  public void initialize(ValidPassword annotation) {
    passwordProperty = annotation.passwordProperty();
    passwordHistoryProperty = annotation.passwordHistoryProperty();
  }

  @Override
  public boolean isValid(Object value, ConstraintValidatorContext context) {
    if (passwordProperty == null) {
      return true;
    }

    String password;

    try {
      Object passwordPropertyValue = PropertyUtils.getProperty(value, passwordProperty);
      password = passwordPropertyValue != null ? passwordPropertyValue.toString() : "";
    } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
      throw new CronappFrameworkException("Error getting password property value", e);
    }

    Collection<String> passwordHistory;

    try {
      Object passwordHistoryValue = PropertyUtils.getProperty(value, passwordHistoryProperty);
      var rawPasswordHistory = passwordHistoryValue != null ? passwordHistoryValue.toString() : null;

      passwordHistory = rawPasswordHistory == null
          ? new ArrayList<>()
          : new ArrayList<>(Arrays.asList(rawPasswordHistory.split(",")));

    } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
      throw new CronappFrameworkException("Error getting password history property value", e);
    }

    if (PasswordEncoder.getPasswordEncoder().isEncoded(password)) {
      return true;
    }

    var rules = new ArrayList<Rule>();

    rules.add(new LengthRule(AppConfig.getMinPasswordSize(), 16));

    rules.add(new CharacterRule(EnglishCharacterData.UpperCase, AppConfig.getMinUpperCase()));
    rules.add(new CharacterRule(EnglishCharacterData.Digit, AppConfig.getMinDigit()));
    rules.add(new CharacterRule(EnglishCharacterData.Special, AppConfig.getMinSpecial()));

    if (AppConfig.getOnlyStrongPass()) {
      rules.add(new IllegalSequenceRule(EnglishSequenceData.Numerical, 3, false));
      rules.add(new IllegalSequenceRule(EnglishSequenceData.Alphabetical, 3, false));
      rules.add(new IllegalSequenceRule(EnglishSequenceData.USQwerty, 3, false));
      rules.add(new WhitespaceRule());
    }

    if (AppConfig.getDontAllowRepeatedPass()) {
      rules.add(new PasswordHistoryRule());
    }

    var passwordData = new PasswordData(password);

    List<PasswordData.Reference> passwordReferences = passwordHistory.stream()
        .map(PasswordData.HistoricalReference::new)
        .collect(Collectors.toUnmodifiableList());

    passwordData.setPasswordReferences(passwordReferences);

    var resourceBundle = Messages.RESOURCE_BUNDLE.get();
    var resolver = resourceBundle == null
        ? new PropertiesMessageResolver()
        : new ResourceBundleMessageResolver(Messages.RESOURCE_BUNDLE.get());

    var validator = new PasswordValidator(resolver, rules);

    var result = validator.validate(passwordData);

    if (result.isValid()) {
      return true;
    }

    context.disableDefaultConstraintViolation();
    validator.getMessages(result).forEach(message -> context.buildConstraintViolationWithTemplate(message).addConstraintViolation());

    return false;
  }
}
