package cronapp.framework.authentication.token;

import cronapi.AppConfig;
import cronapi.RestClient;
import cronapi.Var;
import cronapi.database.DatabaseQueryManager;
import cronapi.database.HistoryListener;
import cronapi.database.TransactionManager;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.authentication.AuthenticationUtil;
import cronapp.framework.authentication.security.CronappAnonymousAuthenticationFilter;
import cronapp.framework.authentication.security.Permission;
import org.eclipse.rap.json.JsonArray;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.vote.UnanimousBased;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;
import org.springframework.security.web.authentication.logout.LogoutSuccessHandler;

import javax.servlet.http.HttpServletResponse;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;

@Configuration
@EnableWebSecurity
@EnableGlobalMethodSecurity(prePostEnabled = true)
public class WebSecurityConfiguration extends WebSecurityConfigurerAdapter {
  
  @Autowired(required = false)
  private Permission permission;
  
  @Autowired
  private EntryPointUnauthorizedHandler unauthorizedHandler;
  
  @Autowired
  private UserDetailsService userDetailsService;

  @Autowired
  private List<AccessDecisionVoter<? extends Object>> decisionVoters;

  @Autowired
  public void configureAuthentication(AuthenticationManagerBuilder authenticationManagerBuilder) throws Exception {
    authenticationManagerBuilder.userDetailsService(userDetailsService).passwordEncoder(passwordEncoder());
  }
  
  @Bean
  public PasswordEncoder passwordEncoder() {
    return new BCryptPasswordEncoder();
  }
  
  @Bean
  public AuthenticationTokenFilter authenticationTokenFilterBean() throws Exception {
    AuthenticationTokenFilter authenticationTokenFilter = new AuthenticationTokenFilter();
    authenticationTokenFilter.setAuthenticationManager(super.authenticationManagerBean());
    return authenticationTokenFilter;
  }
  
  @Override
  protected void configure(AuthenticationManagerBuilder auth) throws Exception {
    auth.userDetailsService(userDetailsService).passwordEncoder(passwordEncoder());
  }
  
  private LogoutSuccessHandler logoutHandler() {
    return (request, response, authentication) -> {
      String authToken = request.getHeader(TokenUtils.AUTH_HEADER_NAME);
      String username = TokenUtils.getUsernameFromToken(authToken);

      if(username != null && SecurityContextHolder.getContext().getAuthentication() == null && TokenUtils.getScopeFromToken(authToken).isEmpty()) {
        if(!TokenUtils.isTokenExpired(authToken)) {
          this.doLogAuthOperation(username);
          if(EventsManager.hasEvent("onLogout")) {
            EventsManager.executeEventOnTransaction("onLogout", Var.valueOf("username", username));
          }
        }
      }

      if(request.getHeader("Accept") == null || !request.getHeader("Accept").contains("json")) {
        response.setStatus(HttpServletResponse.SC_OK);
        response.sendRedirect("/index.html");
      }
    };
  }

  private void doLogAuthOperation(String user){
    // TODO: Centralizar código para evitar duplicidade
    try {
      DatabaseQueryManager logManager = HistoryListener.getAuditLogManager();

      if (logManager != null) {
        String className = logManager.getEntity();
        Class<?> c = Class.forName(className);
        TransactionManager.begin(c);
        org.eclipse.rap.json.JsonObject json = new org.eclipse.rap.json.JsonObject();
        JsonArray arrayParams = new JsonArray();
        json.add("parameters", arrayParams);
        Var auditLog = new Var(new LinkedHashMap<>());
        auditLog.set("type", "app.authorization.Logout");
        auditLog.set("command", "logout");
        auditLog.set("category", "Authorization");
        auditLog.set("date", new Date());
        auditLog.set("objectData", json.toString());
        if (RestClient.getRestClient() != null) {
          auditLog.set("user", user);
          auditLog.set("host", RestClient.getRestClient().getHost());
          auditLog.set("agent", RestClient.getRestClient().getAgent());
        }
        auditLog.set("server", HistoryListener.CURRENT_IP);
        auditLog.set("affectedFields", null);
        auditLog.set("application", AppConfig.guid());
        logManager.insert(auditLog);
        TransactionManager.commit(c);
      }
    }catch (Exception e){
      System.out.print("Error on logging: " + e.getMessage());
    }

  }
  
  @Override
  protected void configure(HttpSecurity httpSecurity) throws Exception {
    // configuration
    httpSecurity.csrf().disable().headers().cacheControl().disable().and().exceptionHandling().authenticationEntryPoint(unauthorizedHandler).and().logout()
            .logoutSuccessHandler(logoutHandler()).and().sessionManagement()
            .sessionCreationPolicy(SessionCreationPolicy.STATELESS);
    // security permission
    if (permission != null) {
      AuthenticationUtil.loadStaticSecurity(httpSecurity);
      permission.loadSecurityPermission(httpSecurity);
    } else {
      httpSecurity.anonymous().authenticationFilter(new CronappAnonymousAuthenticationFilter("anonymousAuthenticationFilterKey", "anonymousUser", ApiManager.getPublicAuthorities()));
      httpSecurity.authorizeRequests().anyRequest().denyAll().accessDecisionManager(new UnanimousBased(decisionVoters));
    }

    httpSecurity.headers().cacheControl().disable()
        .xssProtection().block(false)
        .and().contentTypeOptions()
        .and().httpStrictTransportSecurity().disable();

    // x-frame-options. O AppConfig devolve "SAMEORIGIN" quando não existir
    if(AppConfig.xFrameOptions().equals("SameOrigin")){
      httpSecurity.headers().frameOptions().sameOrigin();
    }else if(AppConfig.xFrameOptions().equals("Deny")){
      httpSecurity.headers().frameOptions().deny();
    }else{
      httpSecurity.headers().frameOptions().disable();
    }

    // Custom JWT based authentication
    httpSecurity.addFilterBefore(authenticationTokenFilterBean(), UsernamePasswordAuthenticationFilter.class);
  }
}