package cronapp.framework.authentication.security;

import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;

import java.time.OffsetDateTime;
import java.util.*;

/**
 * Represents a user in the system
 * The {@link #normalizedUserName} and {@link #normalizedEmail} are there for performance reasons and are used to validate
 * the case insensitive uniqueness of the userName and email fields. They are persisted in the database in order be able
 * to create index on them, thus making the lookups by the normalized user name and email sargable.
 */
public class CronappUserDetails implements UserDetails {
  private String name;
  private String userName;
  private String normalizedUserName;
  private String email;
  private String normalizedEmail;
  private boolean emailConfirmed;
  private String password;
  private String securityStamp;
  private String phoneNumber;
  private boolean phoneNumberConfirmed;
  private boolean twoFactorEnabled;
  private OffsetDateTime lockoutEnd;
  private boolean lockoutEnabled;
  private int accessFailedCount;
  private Set<GrantedAuthority> authorities;
  private byte[] picture;
  private Map<String, Object> payload;

  protected CronappUserDetails() {

  }

  /**
   * @return The user's name.
   */
  public String getName() {
    return name;
  }

  /**
   * @return The user name for this user.
   */
  public String getUserName() {
    return userName;
  }

  /**
   * @return The normalized user name for this user.
   */
  public String getNormalizedUserName() {
    return normalizedUserName;
  }

  /**
   * @return The email address for this user.
   */
  public String getEmail() {
    return email;
  }

  /**
   * @return The normalized email address for this user.
   */
  public String getNormalizedEmail() {
    return normalizedEmail;
  }

  /**
   * @return A flag indicating if a user has confirmed their email address.
   */
  public boolean isEmailConfirmed() {
    return emailConfirmed;
  }

  @Override
  public Collection<? extends GrantedAuthority> getAuthorities() {
    return Collections.unmodifiableSet(authorities);
  }

  /**
   * @return A salted and hashed representation of the password for this user.
   */
  public String getPassword() {
    return password;
  }

  @Override
  public String getUsername() {
    return normalizedUserName;
  }

  @Override
  public boolean isAccountNonExpired() {
    return true;
  }

  @Override
  public boolean isAccountNonLocked() {
    return lockoutEnabled && lockoutEnd != null && lockoutEnd.isBefore(OffsetDateTime.now());
  }

  @Override
  public boolean isCredentialsNonExpired() {
    return true;
  }

  @Override
  public boolean isEnabled() {
    return true;
  }

  /**
   * @return A random value that must change whenever a users credentials change (password changed, login removed)
   */
  public String getSecurityStamp() {
    return securityStamp;
  }

  /**
   * @return A telephone number for the user.
   */
  public String getPhoneNumber() {
    return phoneNumber;
  }

  /**
   * @return A flag indicating if a user has confirmed their telephone address.
   */
  public boolean isPhoneNumberConfirmed() {
    return phoneNumberConfirmed;
  }

  /**
   * @return A flag indicating if two factor authentication is enabled for this user.
   */
  public boolean isTwoFactorEnabled() {
    return twoFactorEnabled;
  }

  /**
   * @return The date and time when any user lockout ends.
   */
  public OffsetDateTime getLockoutEnd() {
    return lockoutEnd;
  }

  /**
   * @return A flag indicating if the user could be locked out.
   */
  public boolean isLockoutEnabled() {
    return lockoutEnabled;
  }

  /**
   * @return The number of failed login attempts for the current user.
   */
  public int getAccessFailedCount() {
    return accessFailedCount;
  }

  public byte[] getPicture() {
    return picture;
  }

  /**
   * @return The payload received from the external authenticator
   */
  public Map<String, Object> getPayload() {
    return payload;
  }

  public static Builder newBuilder() {
    return new Builder();
  }

  public static final class Builder {
    private String name;
    private String userName;
    private String normalizedUserName;
    private String email;
    private String normalizedEmail;
    private boolean emailConfirmed;
    private String password;
    private String securityStamp;
    private String phoneNumber;
    private boolean phoneNumberConfirmed;
    private boolean twoFactorEnabled;
    private OffsetDateTime lockoutEnd;
    private boolean lockoutEnabled;
    private int accessFailedCount;
    private Set<GrantedAuthority> authorities;
    private byte[] picture;
    private Map<String, Object> payload;

    private Builder() {
    }

    public Builder setName(String name) {
      this.name = name;
      return this;
    }

    public Builder setUserName(String userName) {
      this.userName = userName;
      return this;
    }

    public Builder setNormalizedUserName(String normalizedUserName) {
      this.normalizedUserName = normalizedUserName;
      return this;
    }

    public Builder setEmail(String email) {
      this.email = email;
      return this;
    }

    public Builder setNormalizedEmail(String normalizedEmail) {
      this.normalizedEmail = normalizedEmail;
      return this;
    }

    public Builder setEmailConfirmed(boolean emailConfirmed) {
      this.emailConfirmed = emailConfirmed;
      return this;
    }

    public Builder setPassword(String password) {
      this.password = password;
      return this;
    }

    public Builder setSecurityStamp(String securityStamp) {
      this.securityStamp = securityStamp;
      return this;
    }

    public Builder setPhoneNumber(String phoneNumber) {
      this.phoneNumber = phoneNumber;
      return this;
    }

    public Builder setPhoneNumberConfirmed(boolean phoneNumberConfirmed) {
      this.phoneNumberConfirmed = phoneNumberConfirmed;
      return this;
    }

    public Builder setTwoFactorEnabled(boolean twoFactorEnabled) {
      this.twoFactorEnabled = twoFactorEnabled;
      return this;
    }

    public Builder setLockoutEnd(OffsetDateTime lockoutEnd) {
      this.lockoutEnd = lockoutEnd;
      return this;
    }

    public Builder setLockoutEnabled(boolean lockoutEnabled) {
      this.lockoutEnabled = lockoutEnabled;
      return this;
    }

    public Builder setAccessFailedCount(int accessFailedCount) {
      this.accessFailedCount = accessFailedCount;
      return this;
    }

    public Builder setAuthorities(Set<GrantedAuthority> authorities) {
      this.authorities = authorities;
      return this;
    }

    public Builder setPicture(byte[] picture) {
      this.picture = picture;
      return this;
    }

    public Builder setPayload(Map<String, Object> payload) {
      this.payload = payload;
      return this;
    }

    public CronappUserDetails build() {
      CronappUserDetails cronappUserDetails = new CronappUserDetails();
      cronappUserDetails.twoFactorEnabled = this.twoFactorEnabled;
      cronappUserDetails.normalizedEmail = this.normalizedEmail;
      cronappUserDetails.password = this.password;
      cronappUserDetails.securityStamp = this.securityStamp;
      cronappUserDetails.phoneNumberConfirmed = this.phoneNumberConfirmed;
      cronappUserDetails.phoneNumber = this.phoneNumber;
      cronappUserDetails.lockoutEnabled = this.lockoutEnabled;
      cronappUserDetails.emailConfirmed = this.emailConfirmed;
      cronappUserDetails.name = this.name;
      cronappUserDetails.accessFailedCount = this.accessFailedCount;
      cronappUserDetails.lockoutEnd = this.lockoutEnd;
      cronappUserDetails.userName = this.userName;
      cronappUserDetails.authorities = this.authorities;
      cronappUserDetails.normalizedUserName = this.normalizedUserName;
      cronappUserDetails.email = this.email;
      cronappUserDetails.picture = this.picture;
      cronappUserDetails.payload = this.payload;
      return cronappUserDetails;
    }
  }
}
