package cronapp.framework.authentication.saml;

import cronapi.Var;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.authentication.AuthenticationUtil;
import cronapp.framework.authentication.security.CronappAnonymousAuthenticationFilter;
import cronapp.framework.authentication.security.CronappAuthenticationSuccessHandler;
import cronapp.framework.authentication.security.Permission;
import cronapp.framework.authentication.token.AuthenticationTokenFilter;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.lang.Nullable;
import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.vote.UnanimousBased;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.saml.*;
import org.springframework.security.saml.metadata.MetadataDisplayFilter;
import org.springframework.security.saml.metadata.MetadataGenerator;
import org.springframework.security.saml.metadata.MetadataGeneratorFilter;
import org.springframework.security.saml.websso.WebSSOProfileOptions;
import org.springframework.security.web.DefaultSecurityFilterChain;
import org.springframework.security.web.FilterChainProxy;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.access.channel.ChannelProcessingFilter;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.authentication.SimpleUrlAuthenticationFailureHandler;
import org.springframework.security.web.authentication.logout.LogoutHandler;
import org.springframework.security.web.authentication.logout.LogoutSuccessHandler;
import org.springframework.security.web.authentication.logout.SecurityContextLogoutHandler;
import org.springframework.security.web.authentication.www.BasicAuthenticationFilter;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;

import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.List;

@Configuration
@EnableWebSecurity
public class SamlAuthorizationConfigurer extends WebSecurityConfigurerAdapter {

  private final Permission permission;

  private final List<AccessDecisionVoter<? extends Object>> decisionVoters;

  private final ApplicationContext applicationContext;

  public SamlAuthorizationConfigurer(@Nullable Permission permission, List<AccessDecisionVoter<? extends Object>> decisionVoters, ApplicationContext applicationContext) {
    this.permission = permission;
    this.decisionVoters = decisionVoters;
    this.applicationContext = applicationContext;
  }

  @Override
  @Bean
  public AuthenticationManager authenticationManagerBean() throws Exception {
    return super.authenticationManagerBean();
  }

  @Override
  protected void configure(HttpSecurity http) throws Exception {
    // post sem csrf
    http.csrf().disable();

    http
        .httpBasic()
        .authenticationEntryPoint(applicationContext.getBean(SAMLEntryPoint.class));
    http
        .addFilterBefore(applicationContext.getBean(MetadataGeneratorFilter.class), ChannelProcessingFilter.class)
        .addFilterAfter(applicationContext.getBean(FilterChainProxy.class), BasicAuthenticationFilter.class)
        .addFilterAfter(applicationContext.getBean(AuthenticationTokenFilter.class), BasicAuthenticationFilter.class);

    // security permission
    if (permission == null) {
      http.anonymous().authenticationFilter(new CronappAnonymousAuthenticationFilter("anonymousAuthenticationFilterKey", "anonymousUser", ApiManager.getPublicAuthorities()));
      http.authorizeRequests().anyRequest().denyAll().accessDecisionManager(new UnanimousBased(decisionVoters));
    } else {
      permission.loadSecurityPermission(http);
      AuthenticationUtil.loadStaticSecurity(http);
    }

    // x-frame-options disable
    http.headers().cacheControl().disable().frameOptions().disable().httpStrictTransportSecurity().disable();

    http.sessionManagement()
        .sessionCreationPolicy(SessionCreationPolicy.STATELESS);
  }

  @Bean
  public FilterChainProxy samlFilter(SAMLLogoutFilter samlLogoutFilter, MetadataDisplayFilter metadataDisplayFilter, SAMLLogoutProcessingFilter samlLogoutProcessingFilter, SAMLProcessingFilter samlWebSSOProcessingFilter, SAMLWebSSOHoKProcessingFilter samlWebSSOHoKProcessingFilter, SAMLDiscovery samlIDPDiscovery, SAMLEntryPoint samlEntryPoint) {
    List<SecurityFilterChain> chains = new ArrayList<SecurityFilterChain>();
    chains.add(new DefaultSecurityFilterChain(new AntPathRequestMatcher("/saml/login/**"),
        samlEntryPoint));
    chains.add(new DefaultSecurityFilterChain(new AntPathRequestMatcher("/saml/logout/**"),
        samlLogoutFilter));
    chains.add(new DefaultSecurityFilterChain(new AntPathRequestMatcher("/saml/metadata/**"),
        metadataDisplayFilter));
    chains.add(new DefaultSecurityFilterChain(new AntPathRequestMatcher("/saml/SSO/**"),
        samlWebSSOProcessingFilter));
    chains.add(new DefaultSecurityFilterChain(new AntPathRequestMatcher("/saml/SSOHoK/**"),
        samlWebSSOHoKProcessingFilter));
    chains.add(new DefaultSecurityFilterChain(new AntPathRequestMatcher("/saml/SingleLogout/**"),
        samlLogoutProcessingFilter));
    chains.add(new DefaultSecurityFilterChain(new AntPathRequestMatcher("/saml/discovery/**"),
        samlIDPDiscovery));
    return new FilterChainProxy(chains);
  }

  @Bean
  public SAMLDiscovery samlIDPDiscovery() {
    return new SAMLDiscovery();
  }

  @Bean
  public SAMLLogoutProcessingFilter samlLogoutProcessingFilter(LogoutSuccessHandler successLogoutHandler, LogoutHandler logoutHandler) {
    return new SAMLLogoutProcessingFilter(successLogoutHandler, logoutHandler);
  }

  @Bean
  public MetadataDisplayFilter metadataDisplayFilter() {
    return new MetadataDisplayFilter();
  }

  @Bean
  public SAMLProcessingFilter samlWebSSOProcessingFilter(AuthenticationManager authenticationManager, AuthenticationSuccessHandler successHandler, AuthenticationFailureHandler failureHandler) {
    SAMLProcessingFilter samlWebSSOProcessingFilter = new SAMLProcessingFilter();
    samlWebSSOProcessingFilter.setAuthenticationManager(authenticationManager);
    samlWebSSOProcessingFilter.setAuthenticationSuccessHandler(successHandler);
    samlWebSSOProcessingFilter.setAuthenticationFailureHandler(failureHandler);
    return samlWebSSOProcessingFilter;
  }

  @Bean
  public AuthenticationSuccessHandler successHandler() {
    return new CronappAuthenticationSuccessHandler();
  }

  // Handler deciding where to redirect user after failed login
  @Bean
  public SimpleUrlAuthenticationFailureHandler authenticationFailureHandler() {
    SimpleUrlAuthenticationFailureHandler failureHandler = new SimpleUrlAuthenticationFailureHandler();
    failureHandler.setUseForward(true);
    failureHandler.setDefaultFailureUrl("/error");
    return failureHandler;
  }

  @Bean
  public SAMLWebSSOHoKProcessingFilter samlWebSSOHoKProcessingFilter(AuthenticationManager authenticationManager, AuthenticationSuccessHandler successHandler, AuthenticationFailureHandler failureHandler) {
    SAMLWebSSOHoKProcessingFilter samlWebSSOHoKProcessingFilter = new SAMLWebSSOHoKProcessingFilter();
    samlWebSSOHoKProcessingFilter.setAuthenticationSuccessHandler(successHandler);
    samlWebSSOHoKProcessingFilter.setAuthenticationManager(authenticationManager);
    samlWebSSOHoKProcessingFilter.setAuthenticationFailureHandler(failureHandler);
    return samlWebSSOHoKProcessingFilter;
  }

  @Bean
  public SAMLLogoutFilter samlLogoutFilter(LogoutSuccessHandler successLogoutHandler, LogoutHandler logoutHandler) {
    return new SAMLLogoutFilter(successLogoutHandler,
        new LogoutHandler[]{logoutHandler},
        new LogoutHandler[]{logoutHandler});
  }

  @Bean
  public LogoutSuccessHandler successLogoutHandler() {
    return (request, response, authentication) -> {
      if (EventsManager.hasEvent("onLogout") && authentication != null && authentication.getName() != null) {
        EventsManager.executeEventOnTransaction("onLogout", Var.valueOf("username", authentication.getName()));
      }

      if (request.getHeader("Accept") == null || !request.getHeader("Accept").contains("json")) {
        response.setStatus(HttpServletResponse.SC_OK);
        response.sendRedirect("/index.html");
      }
    };
  }

  @Bean
  public SecurityContextLogoutHandler logoutHandler() {
    SecurityContextLogoutHandler logoutHandler =
        new SecurityContextLogoutHandler();
    logoutHandler.setInvalidateHttpSession(true);
    logoutHandler.setClearAuthentication(true);
    return logoutHandler;
  }

  @Bean
  public AuthenticationTokenFilter authenticationTokenFilter(AuthenticationManager authenticationManager) {
    AuthenticationTokenFilter authenticationTokenFilter = new AuthenticationTokenFilter();
    authenticationTokenFilter.setAuthenticationManager(authenticationManager);
    return authenticationTokenFilter;
  }

  @Bean
  public SAMLEntryPoint samlEntryPoint(WebSSOProfileOptions webSSOProfileOptions) {
    SAMLEntryPoint samlEntryPoint = new SAMLEntryPoint();
    samlEntryPoint.setDefaultProfileOptions(webSSOProfileOptions);
    return samlEntryPoint;
  }

  @Bean
  public WebSSOProfileOptions webSSOProfileOptions() {
    WebSSOProfileOptions webSSOProfileOptions = new WebSSOProfileOptions();
    webSSOProfileOptions.setIncludeScoping(false);
    return webSSOProfileOptions;
  }

  @Bean
  public MetadataGeneratorFilter metadataGeneratorFilter(MetadataGenerator metadataGenerator) {
    return new MetadataGeneratorFilter(metadataGenerator);
  }
}
