package cronapp.framework.authentication.normal;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import cronapi.Var;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.authentication.AuthenticationUtil;
import cronapp.framework.authentication.security.CronappAnonymousAuthenticationFilter;
import cronapp.framework.authentication.security.FilterInvocationVoter;
import cronapp.framework.authentication.security.Permission;
import cronapp.framework.i18n.Messages;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpStatus;
import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.vote.UnanimousBased;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.web.authentication.AnonymousAuthenticationFilter;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.authentication.Http403ForbiddenEntryPoint;
import org.springframework.security.web.authentication.logout.LogoutSuccessHandler;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;

import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.util.List;
import java.nio.charset.StandardCharsets;
import java.util.StringJoiner;

/**
 * Classe que configura o WebSecurity, possibilitando requerer que o
 * usuário esteja logado para acessar qualquer usuário
 *
 * @author Techne
 */
@Configuration
@EnableWebSecurity
public class AuthorizationConfigurer extends WebSecurityConfigurerAdapter {

  private static final Logger log = LoggerFactory.getLogger(AuthorizationConfigurer.class);
  
  @Autowired
  private AuthenticationConfigurer authenticationProvider;

  @Autowired(required = false)
  private Permission permission;

  @Autowired
  private List<AccessDecisionVoter<? extends Object>> decisionVoters;


  /**
   * Método que configura o SpringSecurite com o authenticationProvider,
   * responsável também pela criação da base inicial, caso não exista nenhuma
   * permissão cadastrada.
   */
  @Override
  protected void configure(AuthenticationManagerBuilder auth) {
    auth.authenticationProvider(authenticationProvider);
  }

  /**
   * Configurações default para WebSecurity
   */
  @Override
  protected void configure(HttpSecurity http) throws Exception {

    // post sem csrf
    http.csrf().disable();

    // session manager
    http.exceptionHandling()
        .authenticationEntryPoint(new Http403ForbiddenEntryPoint()).and().sessionManagement()
        .maximumSessions(100)
        .maxSessionsPreventsLogin(false)
        .and()
        .sessionCreationPolicy(SessionCreationPolicy.IF_REQUIRED);

    // security permission
    if (permission != null) {
      AuthenticationUtil.loadStaticSecurity(http);
      permission.loadSecurityPermission(http);
    } else {
      http.anonymous().authenticationFilter(new CronappAnonymousAuthenticationFilter("anonymousAuthenticationFilterKey", "anonymousUser", ApiManager.getPublicAuthorities()));
      http.authorizeRequests().anyRequest().denyAll().accessDecisionManager(new UnanimousBased(decisionVoters));
    }

    // login/logout
    http.formLogin()
        .loginProcessingUrl("/auth")
        .successHandler(successHandler())
        .failureHandler(failureHandler())
        .and()
        .logout()
        .logoutRequestMatcher(new AntPathRequestMatcher("/logout"))
        .logoutSuccessHandler(logoutHandler())
        .invalidateHttpSession(true);

    // x-frame-options disable
    http.headers()
        .cacheControl()
        .disable()
        .frameOptions()
        .disable()
        .httpStrictTransportSecurity()
        .disable();
  }

  private LogoutSuccessHandler logoutHandler() {
    return (request, response, authentication) -> {
      if (EventsManager.hasEvent("onLogout") && authentication != null && authentication.getName() != null) {
        EventsManager.executeEventOnTransaction("onLogout", Var.valueOf("username", authentication.getName()));
      }

      if (request.getHeader("Accept") == null || !request.getHeader("Accept").contains("json")) {
        response.setStatus(HttpServletResponse.SC_OK);
        response.sendRedirect("/index.html");
      }
    };
  }

  /**
   * Handler para sucesso de autorização
   */
  @SuppressWarnings("unchecked")
  private AuthenticationSuccessHandler successHandler() {
    return (request, response, authentication) -> {

      HttpSession session = request.getSession();

      User authUser = (User) SecurityContextHolder.getContext().getAuthentication().getPrincipal();

      String username = authUser.getUsername();
      session.setAttribute("username", username);
      session.setAttribute("authorities", authentication.getAuthorities());

      response.setStatus(HttpServletResponse.SC_OK);

      StringJoiner roles = new StringJoiner(",");

      roles.add("Public");
      roles.add("Authenticated");

      boolean root = false;
      for (GrantedAuthority role : authUser.getAuthorities()) {
        roles.add(role.getAuthority());
        if (role.getAuthority().equalsIgnoreCase("Administrators")) {
          root = true;
        }
      }

      try {
        cronapp.framework.api.User user = (cronapp.framework.api.User) request.getAttribute("userDetails");

        Gson gson = new Gson();
        JsonObject json = new JsonObject();
        json.add("user", gson.toJsonTree(user.resetPassword()));
        json.add("picture", gson.toJsonTree(user.getPicture()));
        json.addProperty("roles", roles.toString());
        json.addProperty("theme", "" + session.getAttribute("theme"));
        json.addProperty("root", root);

        response.getOutputStream().write(json.toString().getBytes(StandardCharsets.UTF_8));
        response.setHeader("Content-Type", "application/json;charset=UTF-8");
	  } catch (Exception e) {
        log.error(Messages.getString("AuthError", e.getMessage()), e);
        throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
      }

      if (EventsManager.hasEvent("onLogin")) {
        EventsManager.executeEventOnTransaction("onLogin", Var.valueOf("username", authentication.getName()));
      }
    };
  }

  /**
   * Handler para falha de autorização
   */
  private AuthenticationFailureHandler failureHandler() {
    return (request, response, failureHandler) -> response.setStatus(HttpStatus.UNAUTHORIZED.value());
  }

  @Bean
  public AccessDecisionVoter filterInvocationVoter() {
    return new FilterInvocationVoter();
  }

}
