package cronapp.framework.authentication.external;

import cronapi.AppConfig;
import cronapp.framework.api.ApiManager;
import cronapp.framework.authentication.security.CronappUserDetails;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.ldap.core.DirContextAdapter;
import org.springframework.ldap.core.DirContextOperations;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.SpringSecurityMessageSource;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.ldap.userdetails.UserDetailsContextMapper;

import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import java.time.OffsetDateTime;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

public class ActiveDirectoryUserDetailsMapper implements UserDetailsContextMapper {
  private static final Logger logger = Logger.getLogger(UserDetailsContextMapper.class.getName());
  private final String domain;
  protected MessageSourceAccessor messages = SpringSecurityMessageSource.getAccessor();

  public ActiveDirectoryUserDetailsMapper(String domain) {
    this.domain = domain;
  }

  @Override
  public UserDetails mapUserFromContext(DirContextOperations ctx, String username, Collection<? extends GrantedAuthority> authorities) {
    if (StringUtils.isNotEmpty(AppConfig.groupName())) {
      List<String> groups = Arrays.asList(AppConfig.groupName().split(","));
      if (authorities.stream().noneMatch(authority -> {
        return groups.stream().anyMatch(group -> ApiManager.normalize(authority.getAuthority()).equals(ApiManager.normalize(group)));
      })) {
        throw new BadCredentialsException(messages.getMessage("LdapAuthenticationProvider.badCredentials", "Bad credentials"));
      }
    }

    Attributes attributes = ctx.getAttributes();
    String userName = getAttributeString(attributes, "sAMAccountName");
    String normalizedUserName = ApiManager.normalize(userName);
    String email = getAttributeString(attributes, "mail");

    if (email == null) {
      email = normalizedUserName + "@" + domain;
    }

    String normalizedEmail = ApiManager.normalize(email);

    Set<String> securables = authorities.stream()
        .flatMap(authority -> ApiManager.getRoleSecurables(authority.getAuthority()).stream())
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toSet());

    securables.addAll(ApiManager.getUserSecurables(username).stream()
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toList()));

    securables.addAll(ApiManager.getAuthenticatedSecurables().stream()
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toList()));

    securables.addAll(ApiManager.getPublicSecurables().stream()
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toList()));

    Set<GrantedAuthority> mappedAuthorities = securables.stream().distinct()
        .sorted()
        .map(SimpleGrantedAuthority::new)
        .collect(Collectors.toSet());

    return CronappUserDetails.newBuilder()
        .setName(getAttributeString(attributes, "displayName"))
        .setUserName(userName)
        .setNormalizedUserName(normalizedUserName)
        .setEmail(email)
        .setNormalizedEmail(normalizedEmail)
        .setEmailConfirmed(true)
        .setSecurityStamp(UUID.randomUUID().toString())
        .setPhoneNumber(getAttributeString(attributes, "telephoneNumber"))
        .setPhoneNumberConfirmed(true)
        .setTwoFactorEnabled(false)
        .setLockoutEnd(OffsetDateTime.MIN)
        .setLockoutEnabled(false)
        .setAccessFailedCount(0)
        .setAuthorities(mappedAuthorities)
        .build();
  }

  private static String getAttributeString(Attributes attributes, String name) {
    Object rawAttribute = getAttribute(attributes, name);
    if (rawAttribute == null) {
      return null;
    }

    return rawAttribute.toString();
  }

  private static Object getAttribute(Attributes attributes, String name) {
    try {
      Attribute attr = attributes.get(name);
      if (attr != null) {
        return attr.get();
      }

      return null;
    } catch (NamingException e) {
      logger.log(Level.SEVERE, String.format("Error reading attribute %s", name), e);
      return null;
    }
  }

  @Override
  public void mapUserToContext(UserDetails user, DirContextAdapter ctx) {

  }
}
